/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.utils.binlog;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.primitives.Longs;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.cassandra.concurrent.NamedThreadFactory;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.binlog.BinLogArchiver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExternalArchiver
implements BinLogArchiver {
    private static final Logger logger = LoggerFactory.getLogger(ExternalArchiver.class);
    private static final Pattern PATH = Pattern.compile("%path");
    private static final long DEFAULT_RETRY_DELAY_MS = TimeUnit.MILLISECONDS.convert(5L, TimeUnit.MINUTES);
    private final DelayQueue<DelayFile> archiveQueue = new DelayQueue();
    private final String archiveCommand;
    private final ExecutorService executor = Executors.newSingleThreadExecutor(new NamedThreadFactory("BinLogArchiver"));
    private final Path path;
    private final ExecCommand commandExecutor;
    private volatile boolean shouldContinue = true;

    public ExternalArchiver(String archiveCommand, Path path, int maxArchiveRetries) {
        this(archiveCommand, path, DEFAULT_RETRY_DELAY_MS, maxArchiveRetries, ExternalArchiver::exec);
    }

    @VisibleForTesting
    ExternalArchiver(String archiveCommand, Path path, long retryDelayMs, int maxRetries, ExecCommand command) {
        this.archiveCommand = archiveCommand;
        this.commandExecutor = command;
        this.archiveExisting(path);
        this.path = path;
        this.executor.execute(() -> {
            while (this.shouldContinue) {
                DelayFile toArchive = null;
                try {
                    toArchive = (DelayFile)this.archiveQueue.poll(100L, TimeUnit.MILLISECONDS);
                    if (toArchive == null) continue;
                    this.archiveFile(toArchive.file);
                }
                catch (Throwable t) {
                    if (toArchive != null) {
                        if (toArchive.retries < maxRetries) {
                            logger.error("Got error archiving {}, retrying in {} minutes", new Object[]{toArchive.file, TimeUnit.MINUTES.convert(retryDelayMs, TimeUnit.MILLISECONDS), t});
                            this.archiveQueue.add(new DelayFile(toArchive.file, retryDelayMs, TimeUnit.MILLISECONDS, toArchive.retries + 1));
                            continue;
                        }
                        logger.error("Max retries {} reached for {}, leaving on disk", new Object[]{toArchive.retries, toArchive.file, t});
                        continue;
                    }
                    logger.error("Got error waiting for files to archive", t);
                }
            }
            logger.debug("Exiting archiver thread");
        });
    }

    @Override
    public void onReleased(int cycle, File file) {
        logger.debug("BinLog file released: {}", (Object)file);
        this.archiveQueue.add(new DelayFile(file, 0L, TimeUnit.MILLISECONDS, 0));
    }

    @Override
    public void stop() {
        this.shouldContinue = false;
        try {
            this.executor.submit(() -> {}).get();
            this.archiveExisting(this.path);
        }
        catch (InterruptedException | ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    private void archiveExisting(Path path) {
        if (path == null) {
            return;
        }
        for (File f2 : path.toFile().listFiles(f -> f.isFile() && f.getName().endsWith(".cq4"))) {
            try {
                logger.debug("Archiving existing file {}", (Object)f2);
                this.archiveFile(f2);
            }
            catch (IOException e) {
                logger.error("Got error archiving existing file {}", (Object)f2, (Object)e);
            }
        }
    }

    private void archiveFile(File f) throws IOException {
        String cmd = PATH.matcher(this.archiveCommand).replaceAll(Matcher.quoteReplacement(f.getAbsolutePath()));
        logger.debug("Executing archive command: {}", (Object)cmd);
        this.commandExecutor.exec(cmd);
    }

    static void exec(String command) throws IOException {
        ProcessBuilder pb = new ProcessBuilder(command.split(" "));
        pb.redirectErrorStream(true);
        FBUtilities.exec(pb);
    }

    static interface ExecCommand {
        public void exec(String var1) throws IOException;
    }

    private static class DelayFile
    implements Delayed {
        public final File file;
        private final long delayTime;
        private final int retries;

        public DelayFile(File file, long delay, TimeUnit delayUnit, int retries) {
            this.file = file;
            this.delayTime = System.currentTimeMillis() + TimeUnit.MILLISECONDS.convert(delay, delayUnit);
            this.retries = retries;
        }

        @Override
        public long getDelay(TimeUnit unit) {
            return unit.convert(this.delayTime - System.currentTimeMillis(), TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(Delayed o) {
            DelayFile other = (DelayFile)o;
            return Longs.compare(this.delayTime, other.delayTime);
        }
    }
}

