/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.statistics.distribution;

import java.util.function.DoublePredicate;
import org.apache.commons.statistics.distribution.AbstractDiscreteDistribution;
import org.apache.commons.statistics.distribution.ArgumentUtils;
import org.apache.commons.statistics.distribution.DistributionException;
import org.apache.commons.statistics.distribution.SaddlePointExpansionUtils;

public final class HypergeometricDistribution
extends AbstractDiscreteDistribution {
    private static final double HALF = 0.5;
    private final int numberOfSuccesses;
    private final int populationSize;
    private final int sampleSize;
    private final int lowerBound;
    private final int upperBound;
    private final double bp;
    private final double bq;
    private double[] midpoint;

    private HypergeometricDistribution(int populationSize, int numberOfSuccesses, int sampleSize) {
        this.numberOfSuccesses = numberOfSuccesses;
        this.populationSize = populationSize;
        this.sampleSize = sampleSize;
        this.lowerBound = HypergeometricDistribution.getLowerDomain(populationSize, numberOfSuccesses, sampleSize);
        this.upperBound = HypergeometricDistribution.getUpperDomain(numberOfSuccesses, sampleSize);
        this.bp = (double)sampleSize / (double)populationSize;
        this.bq = (double)(populationSize - sampleSize) / (double)populationSize;
    }

    public static HypergeometricDistribution of(int populationSize, int numberOfSuccesses, int sampleSize) {
        if (populationSize <= 0) {
            throw new DistributionException("Number %s is not greater than 0", populationSize);
        }
        if (numberOfSuccesses < 0) {
            throw new DistributionException("Number %s is negative", numberOfSuccesses);
        }
        if (sampleSize < 0) {
            throw new DistributionException("Number %s is negative", sampleSize);
        }
        if (numberOfSuccesses > populationSize) {
            throw new DistributionException("%s > %s", numberOfSuccesses, populationSize);
        }
        if (sampleSize > populationSize) {
            throw new DistributionException("%s > %s", sampleSize, populationSize);
        }
        return new HypergeometricDistribution(populationSize, numberOfSuccesses, sampleSize);
    }

    private static int getLowerDomain(int nn, int k, int n) {
        return Math.max(0, k - (nn - n));
    }

    private static int getUpperDomain(int k, int n) {
        return Math.min(n, k);
    }

    public int getPopulationSize() {
        return this.populationSize;
    }

    public int getNumberOfSuccesses() {
        return this.numberOfSuccesses;
    }

    public int getSampleSize() {
        return this.sampleSize;
    }

    @Override
    public double probability(int x) {
        return Math.exp(this.logProbability(x));
    }

    @Override
    public double probability(int x0, int x1) {
        if (x0 > x1) {
            throw new DistributionException("Lower bound %s > upper bound %s", x0, x1);
        }
        if (x0 == x1 || x1 < this.lowerBound) {
            return 0.0;
        }
        if (x0 < this.lowerBound) {
            return this.cumulativeProbability(x1);
        }
        if (x1 >= this.upperBound) {
            return this.survivalProbability(x0);
        }
        int lo = x0 + 1;
        int mode = (int)Math.floor(((double)this.sampleSize + 1.0) * ((double)this.numberOfSuccesses + 1.0) / ((double)this.populationSize + 2.0));
        return Math.abs(mode - lo) > Math.abs(mode - x1) ? this.innerCumulativeProbability(lo, x1) : this.innerCumulativeProbability(x1, lo);
    }

    @Override
    public double logProbability(int x) {
        if (x < this.lowerBound || x > this.upperBound) {
            return Double.NEGATIVE_INFINITY;
        }
        return this.computeLogProbability(x);
    }

    private double computeLogProbability(int x) {
        double p1 = SaddlePointExpansionUtils.logBinomialProbability(x, this.numberOfSuccesses, this.bp, this.bq);
        double p2 = SaddlePointExpansionUtils.logBinomialProbability(this.sampleSize - x, this.populationSize - this.numberOfSuccesses, this.bp, this.bq);
        double p3 = SaddlePointExpansionUtils.logBinomialProbability(this.sampleSize, this.populationSize, this.bp, this.bq);
        return p1 + p2 - p3;
    }

    @Override
    public double cumulativeProbability(int x) {
        if (x < this.lowerBound) {
            return 0.0;
        }
        if (x >= this.upperBound) {
            return 1.0;
        }
        double[] mid = this.getMidPoint();
        int m = (int)mid[0];
        if (x < m) {
            return this.innerCumulativeProbability(this.lowerBound, x);
        }
        if (x > m) {
            return 1.0 - this.innerCumulativeProbability(this.upperBound, x + 1);
        }
        return mid[1];
    }

    @Override
    public double survivalProbability(int x) {
        if (x < this.lowerBound) {
            return 1.0;
        }
        if (x >= this.upperBound) {
            return 0.0;
        }
        double[] mid = this.getMidPoint();
        int m = (int)mid[0];
        if (x < m) {
            return 1.0 - this.innerCumulativeProbability(this.lowerBound, x);
        }
        if (x > m) {
            return this.innerCumulativeProbability(this.upperBound, x + 1);
        }
        return 1.0 - mid[1];
    }

    private double innerCumulativeProbability(int x0, int x1) {
        int x = x0;
        double ret = Math.exp(this.computeLogProbability(x));
        if (x0 < x1) {
            while (x != x1) {
                ret += Math.exp(this.computeLogProbability(++x));
            }
        } else {
            while (x != x1) {
                ret += Math.exp(this.computeLogProbability(--x));
            }
        }
        return ret;
    }

    @Override
    public int inverseCumulativeProbability(double p) {
        ArgumentUtils.checkProbability(p);
        return this.computeInverseProbability(p, 1.0 - p, false);
    }

    @Override
    public int inverseSurvivalProbability(double p) {
        ArgumentUtils.checkProbability(p);
        return this.computeInverseProbability(1.0 - p, p, true);
    }

    private int computeInverseProbability(double p, double q, boolean complement) {
        int midPointComparison;
        if (p == 0.0) {
            return this.lowerBound;
        }
        if (q == 0.0) {
            return this.upperBound;
        }
        double[] mid = this.getMidPoint();
        int m = (int)mid[0];
        double mp = mid[1];
        int n = midPointComparison = complement ? Double.compare(1.0 - mp, q) : Double.compare(p, mp);
        if (midPointComparison < 0) {
            return this.inverseLower(p, q, complement);
        }
        if (midPointComparison > 0) {
            return Math.max(m + 1, this.inverseUpper(p, q, complement));
        }
        return m;
    }

    private int inverseLower(double p, double q, boolean complement) {
        int x = this.lowerBound;
        DoublePredicate test = complement ? i -> 1.0 - i > q : i -> i < p;
        double cdf = Math.exp(this.computeLogProbability(x));
        while (test.test(cdf)) {
            cdf += Math.exp(this.computeLogProbability(++x));
        }
        return x;
    }

    private int inverseUpper(double p, double q, boolean complement) {
        int x = this.upperBound;
        DoublePredicate test = complement ? i -> i < q : i -> 1.0 - i > p;
        double sf = 0.0;
        while (test.test(sf)) {
            sf += Math.exp(this.computeLogProbability(x));
            --x;
        }
        if (complement && sf > q || !complement && 1.0 - sf < p) {
            ++x;
        }
        return x;
    }

    @Override
    public double getMean() {
        return (double)this.getSampleSize() * ((double)this.getNumberOfSuccesses() / (double)this.getPopulationSize());
    }

    @Override
    public double getVariance() {
        double N = this.getPopulationSize();
        double K = this.getNumberOfSuccesses();
        double n = this.getSampleSize();
        return n * K * (N - K) * (N - n) / (N * N * (N - 1.0));
    }

    @Override
    public int getSupportLowerBound() {
        return this.lowerBound;
    }

    @Override
    public int getSupportUpperBound() {
        return this.upperBound;
    }

    private double[] getMidPoint() {
        double[] v = this.midpoint;
        if (v == null) {
            double p1;
            int x = this.lowerBound;
            double p0 = 0.0;
            for (p1 = Math.exp(this.computeLogProbability(x)); p1 < 0.5; p1 += Math.exp(this.computeLogProbability(++x))) {
                p0 = p1;
            }
            if (p1 - 0.5 >= 0.5 - p0) {
                --x;
                p1 = p0;
            }
            v = new double[]{x, p1};
            this.midpoint = v;
        }
        return v;
    }
}

