// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

#include "vec/exec/format/parquet/bool_plain_decoder.h"

#include <glog/logging.h>

#include <algorithm>

#include "util/bit_util.h"
#include "vec/columns/column_vector.h"
#include "vec/core/types.h"
#include "vec/exec/format/parquet/parquet_common.h"

namespace doris::vectorized {
#include "common/compile_check_begin.h"
Status BoolPlainDecoder::skip_values(size_t num_values) {
    int skip_cached =
            std::min(num_unpacked_values_ - unpacked_value_idx_, cast_set<int>(num_values));
    unpacked_value_idx_ += skip_cached;
    if (skip_cached == num_values) {
        return Status::OK();
    }
    int num_remaining = cast_set<int>(num_values - skip_cached);
    int num_to_skip = BitUtil::RoundDownToPowerOf2(num_remaining, 32);
    if (num_to_skip > 0) {
        bool_values_.SkipBatch(1, num_to_skip);
    }
    num_remaining -= num_to_skip;
    if (num_remaining > 0) {
        DCHECK_LE(num_remaining, UNPACKED_BUFFER_LEN);
        num_unpacked_values_ =
                bool_values_.UnpackBatch(1, UNPACKED_BUFFER_LEN, &unpacked_values_[0]);
        if (UNLIKELY(num_unpacked_values_ < num_remaining)) {
            return Status::IOError("Can't skip enough booleans in plain decoder");
        }
        unpacked_value_idx_ = num_remaining;
    }
    return Status::OK();
}

Status BoolPlainDecoder::decode_values(MutableColumnPtr& doris_column, DataTypePtr& data_type,
                                       ColumnSelectVector& select_vector, bool is_dict_filter) {
    if (select_vector.has_filter()) {
        return _decode_values<true>(doris_column, data_type, select_vector, is_dict_filter);
    } else {
        return _decode_values<false>(doris_column, data_type, select_vector, is_dict_filter);
    }
}

template <bool has_filter>
Status BoolPlainDecoder::_decode_values(MutableColumnPtr& doris_column, DataTypePtr& data_type,
                                        ColumnSelectVector& select_vector, bool is_dict_filter) {
    auto& column_data = assert_cast<ColumnUInt8*>(doris_column.get())->get_data();
    size_t data_index = column_data.size();
    column_data.resize(data_index + select_vector.num_values() - select_vector.num_filtered());

    ColumnSelectVector::DataReadType read_type;
    while (size_t run_length = select_vector.get_next_run<has_filter>(&read_type)) {
        switch (read_type) {
        case ColumnSelectVector::CONTENT: {
            bool value;
            for (size_t i = 0; i < run_length; ++i) {
                if (UNLIKELY(!_decode_value(&value))) {
                    return Status::IOError("Can't read enough booleans in plain decoder");
                }
                column_data[data_index++] = (UInt8)value;
            }
            break;
        }
        case ColumnSelectVector::NULL_DATA: {
            data_index += run_length;
            break;
        }
        case ColumnSelectVector::FILTERED_CONTENT: {
            bool value;
            for (int i = 0; i < run_length; ++i) {
                if (UNLIKELY(!_decode_value(&value))) {
                    return Status::IOError("Can't read enough booleans in plain decoder");
                }
            }
            break;
        }
        case ColumnSelectVector::FILTERED_NULL: {
            // do nothing
            break;
        }
        }
    }
    return Status::OK();
}
#include "common/compile_check_end.h"

} // namespace doris::vectorized
