/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * license agreements; and to You under the Apache License, version 2.0:
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * This file is part of the Apache Pekko project, which was derived from Akka.
 */

/*
 * Copyright (C) 2009-2022 Lightbend Inc. <https://www.lightbend.com>
 */

package org.apache.pekko.persistence

import scala.concurrent.Future
import scala.concurrent.duration._
import scala.util.control.NoStackTrace

import org.apache.pekko
import pekko.actor._
import pekko.event.Logging
import pekko.event.Logging.Warning
import pekko.persistence.journal.inmem.InmemJournal
import pekko.testkit.{ EventFilter, ImplicitSender, TestEvent }

object EventSourcedActorDeleteFailureSpec {

  case class DeleteTo(n: Long)
  class SimulatedException(msg: String) extends RuntimeException(msg) with NoStackTrace
  class SimulatedSerializationException(msg: String) extends RuntimeException(msg) with NoStackTrace

  class DeleteFailingInmemJournal extends InmemJournal {
    override def asyncDeleteMessagesTo(persistenceId: String, toSequenceNr: Long): Future[Unit] =
      Future.failed(new SimulatedException("Boom! Unable to delete events!"))
  }

  class DoesNotHandleDeleteFailureActor(name: String) extends PersistentActor {
    override def persistenceId = name
    override def receiveCommand: Receive = {
      case DeleteTo(n) => deleteMessages(n)
    }
    override def receiveRecover: Receive = Actor.emptyBehavior
  }

  class HandlesDeleteFailureActor(name: String, probe: ActorRef) extends PersistentActor {
    override def persistenceId = name
    override def receiveCommand: Receive = {
      case DeleteTo(n)              => deleteMessages(n)
      case f: DeleteMessagesFailure => probe ! f
    }
    override def receiveRecover: Receive = Actor.emptyBehavior
  }

}

class EventSourcedActorDeleteFailureSpec
    extends PersistenceSpec(
      PersistenceSpec.config(
        "inmem",
        "SnapshotFailureRobustnessSpec",
        extraConfig = Some(
          """
  pekko.persistence.journal.inmem.class = "org.apache.pekko.persistence.EventSourcedActorDeleteFailureSpec$DeleteFailingInmemJournal"
  """)))
    with ImplicitSender {
  import EventSourcedActorDeleteFailureSpec._

  system.eventStream.publish(TestEvent.Mute(EventFilter[pekko.pattern.AskTimeoutException]()))

  "A persistent actor" must {
    "have default warn logging be triggered, when deletion failed" in {
      val persistentActor = system.actorOf(Props(classOf[DoesNotHandleDeleteFailureActor], name))
      system.eventStream.subscribe(testActor, classOf[Logging.Warning])
      persistentActor ! DeleteTo(Long.MaxValue)
      val message = expectMsgType[Warning].message.toString
      message should include("Failed to deleteMessages")
      message should include("Boom! Unable to delete events!") // the `cause` message
    }

    "be receive an DeleteMessagesFailure when deletion failed, and the default logging should not be triggered" in {
      val persistentActor = system.actorOf(Props(classOf[HandlesDeleteFailureActor], name, testActor))
      system.eventStream.subscribe(testActor, classOf[Logging.Warning])
      persistentActor ! DeleteTo(Long.MaxValue)
      expectMsgType[DeleteMessagesFailure]
      expectNoMessage(100.millis) // since the actor handled the message, we do not issue warn logging automatically
    }

  }
}
