﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/ce/CostExplorerRequest.h>
#include <aws/ce/CostExplorer_EXPORTS.h>
#include <aws/ce/model/DateInterval.h>
#include <aws/ce/model/Expression.h>
#include <aws/ce/model/SortDefinition.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace CostExplorer {
namespace Model {

/**
 */
class GetTagsRequest : public CostExplorerRequest {
 public:
  AWS_COSTEXPLORER_API GetTagsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetTags"; }

  AWS_COSTEXPLORER_API Aws::String SerializePayload() const override;

  AWS_COSTEXPLORER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The value that you want to search for.</p>
   */
  inline const Aws::String& GetSearchString() const { return m_searchString; }
  inline bool SearchStringHasBeenSet() const { return m_searchStringHasBeenSet; }
  template <typename SearchStringT = Aws::String>
  void SetSearchString(SearchStringT&& value) {
    m_searchStringHasBeenSet = true;
    m_searchString = std::forward<SearchStringT>(value);
  }
  template <typename SearchStringT = Aws::String>
  GetTagsRequest& WithSearchString(SearchStringT&& value) {
    SetSearchString(std::forward<SearchStringT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The start and end dates for retrieving the dimension values. The start date
   * is inclusive, but the end date is exclusive. For example, if <code>start</code>
   * is <code>2017-01-01</code> and <code>end</code> is <code>2017-05-01</code>, then
   * the cost and usage data is retrieved from <code>2017-01-01</code> up to and
   * including <code>2017-04-30</code> but not including <code>2017-05-01</code>.</p>
   */
  inline const DateInterval& GetTimePeriod() const { return m_timePeriod; }
  inline bool TimePeriodHasBeenSet() const { return m_timePeriodHasBeenSet; }
  template <typename TimePeriodT = DateInterval>
  void SetTimePeriod(TimePeriodT&& value) {
    m_timePeriodHasBeenSet = true;
    m_timePeriod = std::forward<TimePeriodT>(value);
  }
  template <typename TimePeriodT = DateInterval>
  GetTagsRequest& WithTimePeriod(TimePeriodT&& value) {
    SetTimePeriod(std::forward<TimePeriodT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The key of the tag that you want to return values for.</p>
   */
  inline const Aws::String& GetTagKey() const { return m_tagKey; }
  inline bool TagKeyHasBeenSet() const { return m_tagKeyHasBeenSet; }
  template <typename TagKeyT = Aws::String>
  void SetTagKey(TagKeyT&& value) {
    m_tagKeyHasBeenSet = true;
    m_tagKey = std::forward<TagKeyT>(value);
  }
  template <typename TagKeyT = Aws::String>
  GetTagsRequest& WithTagKey(TagKeyT&& value) {
    SetTagKey(std::forward<TagKeyT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Expression& GetFilter() const { return m_filter; }
  inline bool FilterHasBeenSet() const { return m_filterHasBeenSet; }
  template <typename FilterT = Expression>
  void SetFilter(FilterT&& value) {
    m_filterHasBeenSet = true;
    m_filter = std::forward<FilterT>(value);
  }
  template <typename FilterT = Expression>
  GetTagsRequest& WithFilter(FilterT&& value) {
    SetFilter(std::forward<FilterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The value that you want to sort the data by.</p> <p>The key represents cost
   * and usage metrics. The following values are supported:</p> <ul> <li> <p>
   * <code>BlendedCost</code> </p> </li> <li> <p> <code>UnblendedCost</code> </p>
   * </li> <li> <p> <code>AmortizedCost</code> </p> </li> <li> <p>
   * <code>NetAmortizedCost</code> </p> </li> <li> <p> <code>NetUnblendedCost</code>
   * </p> </li> <li> <p> <code>UsageQuantity</code> </p> </li> <li> <p>
   * <code>NormalizedUsageAmount</code> </p> </li> </ul> <p>The supported values for
   * <code>SortOrder</code> are <code>ASCENDING</code> and
   * <code>DESCENDING</code>.</p> <p>When you use <code>SortBy</code>,
   * <code>NextPageToken</code> and <code>SearchString</code> aren't supported.</p>
   */
  inline const Aws::Vector<SortDefinition>& GetSortBy() const { return m_sortBy; }
  inline bool SortByHasBeenSet() const { return m_sortByHasBeenSet; }
  template <typename SortByT = Aws::Vector<SortDefinition>>
  void SetSortBy(SortByT&& value) {
    m_sortByHasBeenSet = true;
    m_sortBy = std::forward<SortByT>(value);
  }
  template <typename SortByT = Aws::Vector<SortDefinition>>
  GetTagsRequest& WithSortBy(SortByT&& value) {
    SetSortBy(std::forward<SortByT>(value));
    return *this;
  }
  template <typename SortByT = SortDefinition>
  GetTagsRequest& AddSortBy(SortByT&& value) {
    m_sortByHasBeenSet = true;
    m_sortBy.emplace_back(std::forward<SortByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) that uniquely identifies a specific billing
   * view. The ARN is used to specify which particular billing view you want to
   * interact with or retrieve information from when making API calls related to
   * Amazon Web Services Billing and Cost Management features. The BillingViewArn can
   * be retrieved by calling the ListBillingViews API.</p>
   */
  inline const Aws::String& GetBillingViewArn() const { return m_billingViewArn; }
  inline bool BillingViewArnHasBeenSet() const { return m_billingViewArnHasBeenSet; }
  template <typename BillingViewArnT = Aws::String>
  void SetBillingViewArn(BillingViewArnT&& value) {
    m_billingViewArnHasBeenSet = true;
    m_billingViewArn = std::forward<BillingViewArnT>(value);
  }
  template <typename BillingViewArnT = Aws::String>
  GetTagsRequest& WithBillingViewArn(BillingViewArnT&& value) {
    SetBillingViewArn(std::forward<BillingViewArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This field is only used when SortBy is provided in the request. The maximum
   * number of objects that are returned for this request. If MaxResults isn't
   * specified with SortBy, the request returns 1000 results as the default value for
   * this parameter.</p> <p>For <code>GetTags</code>, MaxResults has an upper quota
   * of 1000.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline GetTagsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token to retrieve the next set of results. Amazon Web Services provides
   * the token when the response from a previous call has more results than the
   * maximum page size.</p>
   */
  inline const Aws::String& GetNextPageToken() const { return m_nextPageToken; }
  inline bool NextPageTokenHasBeenSet() const { return m_nextPageTokenHasBeenSet; }
  template <typename NextPageTokenT = Aws::String>
  void SetNextPageToken(NextPageTokenT&& value) {
    m_nextPageTokenHasBeenSet = true;
    m_nextPageToken = std::forward<NextPageTokenT>(value);
  }
  template <typename NextPageTokenT = Aws::String>
  GetTagsRequest& WithNextPageToken(NextPageTokenT&& value) {
    SetNextPageToken(std::forward<NextPageTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_searchString;
  bool m_searchStringHasBeenSet = false;

  DateInterval m_timePeriod;
  bool m_timePeriodHasBeenSet = false;

  Aws::String m_tagKey;
  bool m_tagKeyHasBeenSet = false;

  Expression m_filter;
  bool m_filterHasBeenSet = false;

  Aws::Vector<SortDefinition> m_sortBy;
  bool m_sortByHasBeenSet = false;

  Aws::String m_billingViewArn;
  bool m_billingViewArnHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextPageToken;
  bool m_nextPageTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace CostExplorer
}  // namespace Aws
