--
-- Copyright (c) 2021-2025 Zeping Lee
-- Released under the MIT license.
-- Repository: https://github.com/zepinglee/citeproc-lua
--

local csl_yaml = {}

local yaml
local util

local using_luatex, kpse = pcall(require, "kpse")
if using_luatex then
  yaml = require("tinyyaml")
  util = require("citeproc-util")
else
  yaml = require("tinyyaml")
  util = require("citeproc.util")
end


-- Convert CSL-YAML string to CSL-JSON

---@param str string
---@return CslData
function csl_yaml.parse(str)
  -- lua-tinyyaml 0.4.2 doesn't support dots terminator.
  -- See <https://github.com/api7/lua-tinyyaml/issues/22>.
  str = string.gsub(str, "%.%.%.%s*$", "")

  local status, items = pcall(yaml.parse, str)
  if not status then
    local err = items
    util.error("Failed to parse the YAML contents:\n" .. err)
    return {}
  end

  -- The items exported from Better BibTeX has a {references = []} structure.
  if type(items) == 'table' and items.references then
    items = items.references
  end

  for _, item in ipairs(items) do
    for field, value in pairs(item) do
      if util.variable_types[field] == "date" then
        if type(value) == "number" then
          value = tostring(value)
        end
        if type(value) == "string" then
          -- "2005-11-22"
          -- EDTF is not supported until CSL v1.1.
          local date = util.parse_edtf(value)
          if not date then
            date = {literal = value}
          end
          item[field] = date

        elseif type(value) == "table" and not value["date-parts"] then
          local new_date = {
            ["date-parts"] = {}
          }

          if value.year then
            -- Generated by lua-tinyyaml parser
            new_date["date-parts"][1] = {}
            for j, part_name in ipairs({"year", "month", "day"}) do
              local date_part = value[part_name]
              if date_part then
                new_date["date-parts"][1][j] = date_part
              end
            end

          elseif #value >= 1 then
            -- Exported by Better BibTeX
            for i, range_part in ipairs(value) do
              new_date["date-parts"][i] = {}
              for _, part_name in ipairs({"year", "month", "day"}) do
                local date_part = range_part[part_name]
                if date_part then
                  table.insert(new_date["date-parts"][i], date_part)
                end
              end
            end
          end

          item[field] = new_date
        end
      end
    end
  end

  return items
end


return csl_yaml
