﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/medical-imaging/MedicalImagingRequest.h>
#include <aws/medical-imaging/MedicalImaging_EXPORTS.h>
#include <aws/medical-imaging/model/JobStatus.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace MedicalImaging {
namespace Model {

/**
 */
class ListDICOMImportJobsRequest : public MedicalImagingRequest {
 public:
  AWS_MEDICALIMAGING_API ListDICOMImportJobsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListDICOMImportJobs"; }

  AWS_MEDICALIMAGING_API Aws::String SerializePayload() const override;

  AWS_MEDICALIMAGING_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The data store identifier.</p>
   */
  inline const Aws::String& GetDatastoreId() const { return m_datastoreId; }
  inline bool DatastoreIdHasBeenSet() const { return m_datastoreIdHasBeenSet; }
  template <typename DatastoreIdT = Aws::String>
  void SetDatastoreId(DatastoreIdT&& value) {
    m_datastoreIdHasBeenSet = true;
    m_datastoreId = std::forward<DatastoreIdT>(value);
  }
  template <typename DatastoreIdT = Aws::String>
  ListDICOMImportJobsRequest& WithDatastoreId(DatastoreIdT&& value) {
    SetDatastoreId(std::forward<DatastoreIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The filters for listing import jobs based on status.</p>
   */
  inline JobStatus GetJobStatus() const { return m_jobStatus; }
  inline bool JobStatusHasBeenSet() const { return m_jobStatusHasBeenSet; }
  inline void SetJobStatus(JobStatus value) {
    m_jobStatusHasBeenSet = true;
    m_jobStatus = value;
  }
  inline ListDICOMImportJobsRequest& WithJobStatus(JobStatus value) {
    SetJobStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pagination token used to request the list of import jobs on the next
   * page.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListDICOMImportJobsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The max results count. The upper bound is determined by load testing.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListDICOMImportJobsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_datastoreId;

  JobStatus m_jobStatus{JobStatus::NOT_SET};

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_datastoreIdHasBeenSet = false;
  bool m_jobStatusHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace MedicalImaging
}  // namespace Aws
