#!/usr/bin/perl -w

use IO::File;
use XML::Parser;
use Getopt::Long qw(:config no_ignore_case);
use Encode qw(encode decode);
use File::Basename qw(basename);
use Time::HiRes qw(gettimeofday tv_interval);
use Pod::Usage;


##------------------------------------------------------------------------------
## Constants & Globals
##------------------------------------------------------------------------------
our $prog = basename($0);
our $verbose = 1;     ##-- print progress messages by default

##-- debugging
our $DEBUG = 0;

##-- vars: I/O
our $outfile = "-";   ##-- default: stdout

##-- vars: xml structure
our $c_nameRe  = '^c$';    ##-- element name to index
our $do_xpaths = 1;        ##-- index xpaths?

##-- constants: verbosity levels
our $vl_progress = 1;

##------------------------------------------------------------------------------
## Command-line
##------------------------------------------------------------------------------
GetOptions(##-- General
	   'help|h' => \$help,
	   'verbose|v=i' => \$verbose,
	   'quiet|q' => sub { $verbose=!$_[1]; },

	   ##-- I/O
	   'index|i|char|c=s' => \$c_nameRe,
	   'xpaths|xp|x!' => \$do_xpaths,
	   'output|out|o=s' => \$outfile,
	  );

pod2usage({
	   -exitval=>0,
	   -verbose=>0,
	  }) if ($help);
pod2usage({
	   -message=>"Not enough arguments given!",
	   -exitval=>0,
	   -verbose=>0,
	  }) if (@ARGV < 0); ##-- never


##-- command-line: arguments
push(@ARGV,'-') if (!@ARGV);
our ($srcfile) = @ARGV;


##======================================================================
## Subs: source-file stuff (.chr.xml)

##--------------------------------------------------------------
## XML::Parser handlers

our ($_xp,$_elt,%_attrs);

our $c_re    = qr/$c_nameRe/o;
our $N_c     = 0;     ##-- raw <c> counter
our $N_pb    = 0;     ##-- raw <pb> counter
our $pb_n    = undef; ##-- current pb/@n attribute
our $pb_facs = undef; ##-- current pb/@facs attribute

our @dtrs    = ({});  ##-- daughter-count state stack, ({$dtrName=>$curDtrIndex, ...}, ...)
our @path    = qw();  ##-- element path name stack
our $xpath   = '//*'; ##-- current xpath (default)

## undef = cb_init($expat)
sub cb_init {
  #($_xp) = @_;
  $N_pb = $N_c = 0;
  $pb_n = $pb_facs = undef;
  @dtrs = ({});
  @path = qw();
  $xpath = '//*';
}

## undef = cb_start($expat, $elt,%attrs)
sub cb_start {
  #($_xp,$_elt,%_attrs) = @_;

  ##--------------------------
  $dtrs[$#dtrs]{$_[1]}++;
  push(@dtrs,{});
  push(@path,$_[1]);

  ##--------------------------
  if ($_[1] =~ $c_nameRe) {
    %_attrs = @_[2..$#_];
    $cid = $_attrs{'id'} || $_attrs{'xml:id'};
    $xpath = '/'.join('/', map {$path[$_].'['.$dtrs[$_]{$path[$_]}.']'} (0..$#path)) if ($do_xpaths);
    ##-- dump it
    $outfh->print(join("\t", ($cid||'NULL'), $N_pb, ($pb_n||'NULL'), ($pb_facs||'NULL'), $xpath), "\n");
    ++$N_c;
  }
  ##--------------------------
  elsif ($_[1] eq 'pb') {
    %_attrs = @_[2..$#_];
    ($pb_n,$pb_facs) = @_attrs{'n','facs'};
    ++$N_pb;
  }
}

## undef = cb_end($expat, $elt)
sub cb_end {
  pop(@dtrs);
  pop(@path);
}

## undef = cb_char($expat,$string)
#sub cb_char {}

## undef = cb_default($expat, $str)
#sub cb_default {}

##======================================================================
## MAIN

##-- initialize: @ARGV
push(@ARGV,'-') if (!@ARGV);

##-- initialize output file(s)
$outfile = '-' if (!defined($outfile));
our $outfh = IO::File->new(">$outfile")
  or die("$prog: open failed for output file '$outfile': $!");

##-- initial comments
$outfh->binmode(":utf8");
$outfh->print
  ("%% <(${c_nameRe})>+<pb> index generated by $0\n",
   "%%", ('=' x 70), "\n",
   "%%", join("\t",qw($X_ID $PB_I $PB_N $PB_FACS $X_XPATH),"\n"),
  );


##-- create XML::Parser object
our $xp = XML::Parser->new(
			   ErrorContext => 1,
			   ProtocolEncoding => 'UTF-8',
			   #ParseParamEnt => '???',
			   Handlers => {
					Init   => \&cb_init,
					#XmlDecl => \&cb_xmldecl,
					#Char  => \&cb_char,
					Start  => \&cb_start,
					End    => \&cb_end,
					#Default => \&cb_default,
					#Final   => \&cb_final,
				       },
			  )
  or die("$prog: couldn't create XML::Parser for indexing");
$xp->parsefile($srcfile);
print STDERR ("$prog: indexed $N_c ($c_nameRe) elements, $N_pb <pb>s: ", sprintf("%.1f", $N_c/($N_pb||1)), " elts/page\n")
  if ($verbose>=$vl_progress);

##-- wnd

__END__

=pod

=head1 NAME

dtatw-mkpx.perl - make pagebreak index for a DTA source XML file

=head1 SYNOPSIS

 dtatw-mkpx.perl [OPTIONS] C_XML_FILE

 General Options:
  -help                  # this help message
  -verbose LEVEL         # set verbosity level (0<=LEVEL<=1)
  -quiet                 # be silent

 I/O Options:
  -index  REGEX          # target element regex (default='^c$')
  -output FILE           # specify output file (default='-' (STDOUT))

=cut

##------------------------------------------------------------------------------
## Options and Arguments
##------------------------------------------------------------------------------
=pod

=head1 OPTIONS AND ARGUMENTS

Not yet written.

=cut

##------------------------------------------------------------------------------
## Description
##------------------------------------------------------------------------------
=pod

=head1 DESCRIPTION

Make a pagebreak index for a DTA source XML file.

=cut

##------------------------------------------------------------------------------
## See Also
##------------------------------------------------------------------------------
=pod

=head1 SEE ALSO

L<dtatw-add-c.perl(1)|dtatw-add-c.perl>,
L<dta-tokwrap.perl(1)|dta-tokwrap.perl>,
L<dtatw-add-ws.perl(1)|dtatw-add-ws.perl>,
L<dtatw-rm-c.perl(1)|dtatw-rm-c.perl>,
...

=cut

##------------------------------------------------------------------------------
## Footer
##------------------------------------------------------------------------------
=pod

=head1 AUTHOR

Bryan Jurish E<lt>moocow@cpan.orgE<gt>

=cut
