package MToken::DeviceSkel; # $Id: DeviceSkel.pm 101 2021-10-09 18:57:43Z minus $
use strict;
use utf8;

=encoding utf8

=head1 NAME

MToken::DeviceSkel - Internal helper's methods for MToken

=head1 VIRSION

Version 1.01

=head1 SYNOPSIS

none

=head1 DESCRIPTION

Internal helper's methods for MToken

no public methods

=head2 build, dirs, pool

Main methods. For internal use only

=head1 SEE ALSO

L<MToken>, L<CTK::Skel>

=head1 AUTHOR

Serż Minus (Sergey Lepenkov) L<http://www.serzik.com> E<lt>abalama@cpan.orgE<gt>

=head1 COPYRIGHT

Copyright (C) 1998-2021 D&D Corporation. All Rights Reserved

=head1 LICENSE

This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

See C<LICENSE> file and L<https://dev.perl.org/licenses/>

=cut

use constant SIGNATURE => "device";

use vars qw/ $VERSION /;
$VERSION = '1.01';

use MToken::Const qw/DIR_PRIVATE DEVICE_CONF_FILE DEVICE_MANIFEST_FILE DB_FILE/;

sub build { # Building
    my $self = shift;

    my $rplc = $self->{rplc};
    $rplc->{TODO} = "TODO";
    $rplc->{ENDSIGN} = "__END__"; # END signature
    $rplc->{DIR_PRIVATE} = DIR_PRIVATE;
    $rplc->{DEVICE_CONF_FILE} = DEVICE_CONF_FILE;
    $rplc->{DEVICE_MANIFEST_FILE} = DEVICE_MANIFEST_FILE;
    $rplc->{DB_FILE} = DB_FILE;

    $self->maybe::next::method();
    return 1;
}
sub dirs { # Directories and permissions as array of hashs
    my $self = shift;
    $self->{subdirs}{(SIGNATURE)} = [
        {
            path => DIR_PRIVATE,
            mode => 0755,
        },
    ];
    $self->maybe::next::method();
    return 1;
}
sub pool { # Multipart pool of files
    my $self = shift;
    my $pos =  tell DATA;
    my $data = scalar(do { local $/; <DATA> });
    seek DATA, $pos, 0;
    $self->{pools}{(SIGNATURE)} = $data;
    $self->maybe::next::method();
    return 1;
}

1;

__DATA__

-----BEGIN FILE-----
Name: DESCRIPTION
File: DESCRIPTION
Mode: 666

################################################################################
##
## MTOKEN      : %TOKEN_NAME%
## GENERATED   : %GMT%
## PACKAGE     : %PACKAGE% %VERSION%
##
################################################################################

This token device (folder or storage) holds keys and tokens for access to
various systems, web-sites, programs and etc.

-----END FILE-----

-----BEGIN FILE-----
Name: README
File: README
Mode: 666

This token device (folder or storage) holds keys and tokens for access to
various systems, web-sites, programs and etc.

-----END FILE-----

-----BEGIN FILE-----
Name: %DEVICE_CONF_FILE%
File: %DIR_PRIVATE%/%DEVICE_CONF_FILE%
Mode: 666

#
# This file contains data for device configuration (dconfig)
# Local configuration holda in "%DIR_PRIVATE%" directory of
# your home directory (lconfig)
# See also the global config file (gconfig) that is located in the
# system config directory (/etc/mtoken)
#

## General
TOKEN       %TOKEN_NAME%
SERVER_URL  %SERVER_URL%

-----END FILE-----

-----BEGIN FILE-----
Name: %DEVICE_CONF_FILE%
File: %DIR_PRIVATE%/%DEVICE_MANIFEST_FILE%
Mode: 666

%DIR_PRIVATE%/%DEVICE_MANIFEST_FILE%
%DIR_PRIVATE%/%DEVICE_CONF_FILE%
%DIR_PRIVATE%/%DB_FILE%
README
DESCRIPTION
-----END FILE-----

-----BEGIN FILE-----
Name: LICENSE
File: LICENSE
Mode: 644
Encode: base64
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-----END FILE-----
