/**
 * Manages batch updates, error handling, and e-mail notifications.
 * 
 * @author Zlatan Hot
 * @since 19/12/2014
 * 
 */
global abstract class AbstractBatch implements Database.Batchable<sObject>, Database.Stateful {

    // batch query
    protected String query;

    // error categories
    Errors errors;
    // messages
    protected List<String> messages;

    // job start time
    protected DateTime jobStartTime;
    // batch start time
    protected DateTime batchStartTime;

    // current batch number
    protected Integer batchCount;
    // number of updated records
    protected Integer recordCount;

    // validation bypass setting
    Validation_Bypass__c validationBypass;
    Boolean validationBypassInserted;
    // workflow bypass setting
    Workflow_Bypass__c workflowBypass;
    Boolean workflowBypassInserted;

    global AbstractBatch() {
        query = '';
        errors = new Errors();
        messages = new List<String>();
        batchCount = 0;
        recordCount = 0;
    }

	global virtual Database.QueryLocator start(Database.BatchableContext batchableContext) {
        // initialize the job start time
        if (jobStartTime == null) {
            jobStartTime = DateTime.now();
        }
		return Database.getQueryLocator(query);
	}

   	global virtual void execute(Database.BatchableContext batchableContext, List<sObject> scope) {
        try {
            // debug info
            if (batchStartTime == null) {
                messages.add('Preparation lasted: ' + utl_Time.getElapsedTime(jobStartTime, DateTime.now()));
            }
            batchCount ++;
            batchStartTime = DateTime.now();

            // add execution logic here
            // ...

        } catch (Exception e) {
            sendMail(batchableContext.getJobId());
            throw e;
        }
	}

	global virtual void finish(Database.BatchableContext batchableContext) {
        sendMail(batchableContext.getJobId());
	}

    /**
    * Partially updates specified records.
    * The opt_allOrNone parameter is set to false.
    * Temporarily disables triggers, validation, and workflows while performing the DML operation.
    * Records that cannot be updated are logged.
    * @param records records that should be updated
    * @return number of succesfully updated records
    * 
    */
    protected Integer updateRecords(List<sObject> records) {
        Integer numberOfUpdatedRecords = records.size();
        disableTriggersAndValidationAndWorkflows();
        Database.SaveResult[] results = database.update(records, false);
        enableTriggersAndValidationAndWorkflows();
        // log errors
        for (Integer i = 0; i < results.size(); i++) {
            Database.SaveResult result = results[i];
            if (!result.isSuccess()) {
                numberOfUpdatedRecords --;
                for (Database.Error error : result.getErrors()) {
                    errors.log('' + error.getStatusCode(), records[i].Id);
                }
            }
        }
        return numberOfUpdatedRecords;
    }

    /**
    * Sends an e-mail with job execution details.
    * @param jobId AsyncApexJob Id
    * 
    */    
    protected void sendMail(Id jobId) {
		AsyncApexJob job = [SELECT Id, ApexClass.Name, Status, TotalJobItems, JobItemsProcessed, NumberOfErrors, CreatedBy.Email
                            FROM AsyncApexJob
                            WHERE Id = :jobId];
        Messaging.SingleEmailMessage mail = new Messaging.SingleEmailMessage();
        mail.setSubject(UserInfo.getOrganizationName() + ': Job ' + job.ApexClass.Name + ' ' + job.Status);
        mail.setToAddresses(new List<String> { job.CreatedBy.Email });
        mail.setReplyTo('no-reply@sophos.com');
        mail.setSenderDisplayName(job.ApexClass.Name);
        String html = 
            '<table style="border-collapse: collapse; border: 1px solid #344a5f;">' +
                '<thead style="background-color: #2a94d6; color: white; font-weight: bold;">' + 
                    '<tr>' + 
            			'<th style="padding: 5px; border: 1px solid #344a5f;">Job ID</th>' + 
            			'<th style="padding: 5px; border: 1px solid #344a5f;">Apex Class</th>' + 
            			'<th style="padding: 5px; border: 1px solid #344a5f;">Status</th>' + 
            			'<th style="padding: 5px; border: 1px solid #344a5f;">Total Batches</th>' + 
            			'<th style="padding: 5px; border: 1px solid #344a5f;">Batches Processed</th>' +
            			'<th style="padding: 5px; border: 1px solid #344a5f;">Failures</th>' + 
            			'<th style="padding: 5px; border: 1px solid #344a5f;">Start</th>' + 
            			'<th style="padding: 5px; border: 1px solid #344a5f;">End</th>' + 
            			'<th style="padding: 5px; border: 1px solid #344a5f;">Duration</th>' + 
            		'</tr>' +
                '</thead>' +
                '<tbody>' +
                    '<tr style="background-color: #f0f1f2; text-align: center;">' + 
            			'<td style="padding: 5px; border: 1px solid #344a5f;">' + job.Id + '</td>' + 
            			'<td style="padding: 5px; border: 1px solid #344a5f;">' + job.ApexClass.Name + '</td>' + 
            			'<td style="padding: 5px; border: 1px solid #344a5f;">' + job.Status + '</td>' + 
            			'<td style="padding: 5px; border: 1px solid #344a5f;">' + job.TotalJobItems + '</td>' + 
            			'<td style="padding: 5px; border: 1px solid #344a5f;">' + job.JobItemsProcessed + '</td>' + 
            			'<td style="padding: 5px; border: 1px solid #344a5f;">' + job.NumberOfErrors + '</td>' + 
            			'<td style="padding: 5px; border: 1px solid #344a5f;">' + jobStartTime.format() + '</td>' + 
            			'<td style="padding: 5px; border: 1px solid #344a5f;">' + DateTime.now().format() + '</td>' + 
            			'<td style="padding: 5px; border: 1px solid #344a5f;">' + utl_Time.getElapsedTime(jobStartTime, DateTime.now()) + '</td>' + 
            		'</tr>' +
                '</tbody>' + 
            '</table>';
        html += '<div style="background-color: #4ab471; border-top-left-radius: 3px; border-top-right-radius: 3px; color: white; font-weight: bold; padding: 5px; margin-top: 15px;">Messages</div>';
        html += '<div style="background-color: #f0f1f2; border-bottom-left-radius: 3px; border-bottom-right-radius: 3px; padding: 2px;">';
        for (String message : messages) {
            html += '<div style="padding: 5px">' + message + '</div>';
        }
        html += '</div>';
        html += '<div style="background-color: #cf5c60; border-top-left-radius: 3px; border-top-right-radius: 3px; color: white; font-weight: bold; padding: 5px; margin-top: 15px;">Errors</div>';
        html += '<div style="background-color: #f0f1f2; border-bottom-left-radius: 3px; border-bottom-right-radius: 3px; padding: 2px;">';
        for (String category : errors.getCategories()) {
            List<Object> items = errors.getItems(category);
            html += '<div style="color:#cf5c60; font-weight: bold; padding: 5px">' + category + ' (' + items.size() + ')</div><div style="padding-left: 15px">';
            for (Object item: items) {
            	html += '<div>' + item + '</div>';
            }
            html += '</div>';
        }
        html += '</div>';

        mail.setHtmlBody(html);
        Messaging.sendEmail(new Messaging.SingleEmailMessage[] { mail });
    }

    /**
    * Disables triggers, validation, and workflows.
    * Used right before performing an expensive DML operation.
    * 
    */ 
    protected void disableTriggersAndValidationAndWorkflows() {
        utl_User.isTriggerDeactivationEnabledForAllTriggers = true;
        disableValidation();
        disableWorkflows();
    }

    /**
    * Enables triggers, validation, and workflows.
    * Used to undo disabling the above mentioned, right after performing an expensive DML operation.
    * 
    */     
    protected void enableTriggersAndValidationAndWorkflows() {
        utl_User.isTriggerDeactivationEnabledForAllTriggers = false;
        enableValidation();
        enableWorkflows();
    }

    /**
    * Disables workflows using workflow bypass for the current user.
    * 
    */     
    protected void disableWorkflows() {
        workflowBypass = Workflow_Bypass__c.getInstance(UserInfo.getUserId());
        if (workflowBypass == null || workflowBypass.Id == null) {
            workflowBypass = new Workflow_Bypass__c(SetupOwnerId = Userinfo.getUserId(), Switch_off_outbound_emails__c = true);
            insert workflowBypass;
            workflowBypassInserted = true;
        } else {
            workflowBypass.Switch_off_outbound_emails__c = true;
            workflowBypass.Switch_off_outbound_messages__c = true;
            update workflowBypass;
            workflowBypassInserted = false;
        }
    }

    /**
    * Enables workflows using workflow bypass for the current user.
    * 
    */       
    protected void enableWorkflows() {
        if (workflowBypassInserted) {
            delete workflowBypass;
        }
        else {
            workflowBypass.Switch_off_outbound_emails__c = false;
            workflowBypass.Switch_off_outbound_messages__c = false;
            update workflowBypass;
        }
    }

    /**
    * Disables validation using validation bypass for the current user.
    * 
    */  
    protected void disableValidation() {
        validationBypass = Validation_Bypass__c.getInstance(UserInfo.getUserId());
        if (validationBypass == null || validationBypass.Id == null) {
            validationBypass = new Validation_Bypass__c(SetupOwnerId = Userinfo.getUserId(), Level_1__c = true, Level_2__c = true);
            insert validationBypass;
            validationBypassInserted = true;
        } else {
            validationBypass.Level_1__c = true;
            validationBypass.Level_2__c = true;
            update validationBypass;
            validationBypassInserted = false;
        }
    }

    /**
    * Enables validation using validation bypass for the current user.
    * 
    */      
    protected void enableValidation() {
        if (validationBypassInserted) {
            delete validationBypass;
        }
        else {
            validationBypass.Level_1__c = false;
            validationBypass.Level_2__c = false;
            update validationBypass;
        }
    }

    /**
    * Manages error categories and items
    * 
    */      
    public class Errors {
        // error items by category
        Map<String, List<Object>> errors = new Map<String, List<Object>>();

        public Set<String> getCategories() {
            return errors.keySet();
        }

        public List<Object> getItems(String category) {
            return errors.get(category);
        }

        public Integer getCategorySize(String category) {
            List<Object> items = getItems(category);
            return items != null ? items.size() : 0;
        }

        public void log(String category, Object item) {
            List<Object> items = getItems(category);
            if (items == null) {
                items = new List<Object>();
            }
            items.add(item);
            errors.put(category, items);
        }
    }

}