package Business::MaxMind::TelephoneVerification;

use strict;

use LWP::UserAgent;
use base 'Business::MaxMind::HTTPBase';

our $VERSION = '1.55';

my @allowed_fields = qw/l phone verify_code language/;

sub _init {
    my $self = shift;
    $self->{url}         = 'app/telephone_http';
    $self->{check_field} = 'refid';
    $self->{timeout} ||= 30
        ; # provide a default value of 10 seconds for timeout if not set by user
    %{ $self->{allowed_fields} } = map { $_ => 1 } @allowed_fields;
}

1;

=pod

=encoding UTF-8

=head1 NAME

Business::MaxMind::TelephoneVerification - Access MaxMind's Telephone Verification services

=head1 VERSION

version 1.55

=head1 DESCRIPTION

This module queries the Telephone Verification service and calls the customer

=over

=item new

Class method that returns a Business::MaxMind::TelephoneVerification object.  If isSecure is set to 1, it will use a
secure connection.  If debug is set to 1, will print debugging output to standard error.  timeout parameter is used
to set timeout in seconds, if absent default value for timeout is 10 seconds.

=item input

Sets input fields.  The input fields are

=for html <ul>
  <li><b>l:</b> license key
  <li><b>phone:</b> phone number
  <li><b>verify_code:</b> code to used to verify
</ul>

Returns 1 on success, 0 on failure.

=item query

Sends out query to MaxMind server and waits for response.  If the primary
server fails to respond, it sends out a request to the secondary server.

=item output

Returns the output returned by the MaxMind server as a hash reference.
The output fields are

=for html <ul>
  <li><b>refid:</b> Reference ID
  <li><b>randCode:</b> code to used to verify, randomly generated by MaxMind only if not passed as verify_code input parameter
  <li><b>err:</b> Error String
</ul>

=back

=head1 SEE ALSO

L<https://www.maxmind.com/en/telephone_overview>

=head1 AUTHORS

=over 4

=item *

TJ Mather <tjmather@maxmind.com>

=item *

Frank Mather <frank@maxmind.com>

=back

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2014 by MaxMind, Inc..

This is free software, licensed under:

  The GNU General Public License, Version 2, June 1991

=cut

__END__

# ABSTRACT: Access MaxMind's Telephone Verification services

