package XML::CompareML::HTML;
$XML::CompareML::HTML::VERSION = '0.2.13';
use strict;
use warnings;

use Carp;
use File::Spec;

use CGI ();
use XML::LibXSLT;

use XML::CompareML::ConfigData;

use parent 'XML::CompareML::Base';

__PACKAGE__->mk_accessors(
    qw(
        _data_dir
        _xml_parser
        _stylesheet
        )
);

sub _initialize
{
    my $self = shift;

    $self->SUPER::_initialize(@_);

    my (%args) = (@_);

    my $data_dir = $args{'data_dir'}
        || XML::CompareML::ConfigData->config('extradata_install_path')->[0];

    $self->_data_dir($data_dir);

    $self->_xml_parser( XML::LibXML->new() );

    my $xslt = XML::LibXSLT->new();

    my $style_doc =
        $self->_xml_parser()
        ->parse_file(
        File::Spec->catfile( $self->_data_dir(), "compare-ml.xslt" ),
        );

    $self->_stylesheet( $xslt->parse_stylesheet($style_doc) );

    return 0;
}


sub process
{
    my ( $self, $args ) = @_;


    my $stylesheet = $self->_stylesheet();

    my $results = $stylesheet->transform( $self->dom() );

    print { *{ $self->{o} } } $stylesheet->output_string($results);

    return 0;
}



sub gen_systems_list
{
    my ( $self, %args ) = @_;

    my $fh = $args{output_handle};

    my @implementations =
        $self->_findnodes("/comparison/meta/implementations/impl");

    foreach my $impl (@implementations)
    {
        my $name     = $self->_impl_get_tag_text( $impl, "name" );
        my $url      = $self->_impl_get_tag_text( $impl, "url" );
        my $fullname = $self->_impl_get_tag_text( $impl, "fullname" );
        my $vendor   = $self->_impl_get_tag_text( $impl, "vendor" );
        if ( !defined($url) )
        {
            die "URL not specified for implementation $name.";
        }
        print {$fh} qq{<li><a href="}
            . CGI::escapeHTML($url) . qq{">}
            . CGI::escapeHTML( defined($fullname) ? $fullname : $name )
            . qq{</a>}
            . ( defined($vendor) ? " by $vendor" : "" )
            . qq{</li>\n};
    }
}


1;

__END__

=pod

=encoding UTF-8

=head1 NAME

XML::CompareML::HTML - convert CompareML to XHTML

=head1 VERSION

version 0.2.13

=head1 SYNOPSIS

See L<XML::CompareXML>.

=head2 $to-html->process()

Do the actual processing using the XSLT stylesheet.

=begin RELAX_NG_VALIDATION

    my $ret_code;

    eval
    {
        $ret_code = $self->_rng()->validate($source_dom);
    };

    if (defined($ret_code) && ($ret_code == 0))
    {
        # It's OK.
    }
    else
    {
        confess "RelaxNG validation failed [\$ret_code == $ret_code ; $@]";
    }

=end RELAX_NG_VALIDATION

=head1 METHODS

=head2 $converter->gen_systems_list({output_handle => \*STDOUT})

Generates a list of li's with links to the systems, not unlike:

L<http://better-scm.berlios.de/comparison/>

=head1 AUTHOR

Shlomi Fish, L<http://www.shlomifish.org/>.

=head1 SEE ALSO

L<XML::CompareML>

=head1 COPYRIGHT AND LICENSE

Copyright 2004, Shlomi Fish. All rights reserved.

You can use, modify and distribute this module under the terms of the MIT X11
license. ( L<http://www.opensource.org/licenses/mit-license.php> ).

=for :stopwords cpan testmatrix url bugtracker rt cpants kwalitee diff irc mailto metadata placeholders metacpan

=head1 SUPPORT

=head2 Websites

The following websites have more information about this module, and may be of help to you. As always,
in addition to those websites please use your favorite search engine to discover more resources.

=over 4

=item *

MetaCPAN

A modern, open-source CPAN search engine, useful to view POD in HTML format.

L<https://metacpan.org/release/XML-CompareML>

=item *

RT: CPAN's Bug Tracker

The RT ( Request Tracker ) website is the default bug/issue tracking system for CPAN.

L<https://rt.cpan.org/Public/Dist/Display.html?Name=XML-CompareML>

=item *

CPANTS

The CPANTS is a website that analyzes the Kwalitee ( code metrics ) of a distribution.

L<http://cpants.cpanauthors.org/dist/XML-CompareML>

=item *

CPAN Testers

The CPAN Testers is a network of smoke testers who run automated tests on uploaded CPAN distributions.

L<http://www.cpantesters.org/distro/X/XML-CompareML>

=item *

CPAN Testers Matrix

The CPAN Testers Matrix is a website that provides a visual overview of the test results for a distribution on various Perls/platforms.

L<http://matrix.cpantesters.org/?dist=XML-CompareML>

=item *

CPAN Testers Dependencies

The CPAN Testers Dependencies is a website that shows a chart of the test results of all dependencies for a distribution.

L<http://deps.cpantesters.org/?module=XML::CompareML>

=back

=head2 Bugs / Feature Requests

Please report any bugs or feature requests by email to C<bug-xml-compareml at rt.cpan.org>, or through
the web interface at L<https://rt.cpan.org/Public/Bug/Report.html?Queue=XML-CompareML>. You will be automatically notified of any
progress on the request by the system.

=head2 Source Code

The code is open to the world, and available for you to hack on. Please feel free to browse it and play
with it, or whatever. If you want to contribute patches, please send me a diff or prod me to pull
from your repository :)

L<https://github.com/shlomif/xml-compareml>

  git clone git://github.com/shlomif/xml-compareml.git

=head1 AUTHOR

Shlomi Fish <shlomif@cpan.org>

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website
L<https://github.com/shlomif/xml-compareml/issues>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2018 by Shlomi Fish.

This is free software, licensed under:

  The MIT (X11) License

=cut
