<?php
declare( strict_types = 1 );

namespace Wikimedia\Parsoid\NodeData;

use Psr\Container\ContainerInterface;
use stdClass;
use Wikimedia\JsonCodec\Hint;
use Wikimedia\JsonCodec\JsonClassCodec;
use Wikimedia\JsonCodec\JsonCodecable;
use Wikimedia\JsonCodec\JsonCodecInterface;
use Wikimedia\Parsoid\DOM\DocumentFragment;
use Wikimedia\Parsoid\Tokens\SourceRange;
use Wikimedia\Parsoid\Tokens\Token;
use Wikimedia\Parsoid\Utils\DOMDataUtils;

/**
 * Editing data for a DOM node.  Managed by DOMDataUtils::get/setDataMw().
 *
 * To reduce memory usage, most of the properties need to be dynamic, but
 * we use the property declarations below to allow type checking.
 *
 * @property list<string|TemplateInfo> $parts
 * @property string $name
 * @property string $extPrefix
 * @property string $extSuffix
 * @property list<DataMwAttrib> $attribs Extended attributes of an HTML tag
 * @property string $src
 * @property DocumentFragment $caption
 * @property string $thumb
 * @property bool $autoGenerated
 * @property list<DataMwError> $errors
 * @property DataMwBody $body
 * @property mixed $html
 * @property float $scale
 * @property string $starttime
 * @property string $endtime
 * @property string $thumbtime
 * @property string $page
 * == Annotations ==
 * @property string $rangeId
 * @property SourceRange $wtOffsets
 * @property bool $extendedRange
 * @property stdClass $attrs Attributes for an extension tag or annotation (T367616 should be renamed)
 */
#[\AllowDynamicProperties]
class DataMw implements JsonCodecable {

	public function __construct( array $initialVals = [] ) {
		foreach ( $initialVals as $k => $v ) {
			switch ( $k ) {
				case 'attrs':
					// T367616: facilitate renaming.
					$this->attrs = $v;
					break;
				// Add cases here for components which should be instantiated
				// as proper classes.
				default:
					$this->$k = $v;
					break;
			}
		}
	}

	/** Returns true iff there are no dynamic properties of this object. */
	public function isEmpty(): bool {
		$result = (array)$this;
		return $result === [];
	}

	/**
	 * Helper method to facilitate renaming the 'attrs' property to
	 * 'extAttribs' (T367616).
	 * @note that numeric key values will be converted from string
	 *   to int by PHP when they are used as array keys
	 * @return ?array<string|int,string|array<Token|string>>
	 */
	public function getExtAttribs(): ?array {
		if ( isset( $this->attrs ) ) {
			return (array)$this->attrs;
		}
		return null;
	}

	/**
	 * Helper method to facilitate renaming the 'attrs' property to
	 * 'extAttribs' (T367616).
	 * @param string $name
	 * @return string|array<Token|string>|null
	 */
	public function getExtAttrib( string $name ) {
		return $this->attrs->$name ?? null;
	}

	/**
	 * Helper method to facilitate renaming the 'attrs' property to
	 * 'extAttribs' (T367616).
	 * @param string $name
	 * @param string|array<Token|string>|null $value
	 *  Setting to null will unset it from the array.
	 */
	public function setExtAttrib( string $name, $value ): void {
		if ( !isset( $this->attrs ) ) {
			$this->attrs = (object)[];
		}
		if ( $value === null ) {
			unset( $this->attrs->$name );
		} else {
			$this->attrs->$name = $value;
		}
	}

	public function __clone() {
		// Deep clone non-primitive properties
		if ( isset( $this->parts ) ) {
			foreach ( $this->parts as &$part ) {
				if ( !is_string( $part ) ) {
					$part = clone $part;
				}
			}
		}
		// Properties which are lists of cloneable objects
		foreach ( [ 'attribs', 'errors' ] as $prop ) {
			if ( isset( $this->$prop ) ) {
				foreach ( $this->$prop as &$item ) {
					$item = clone $item;
				}
			}
		}
		// Properties which are cloneable objects
		foreach ( [ 'body', 'wtOffsets' ] as $prop ) {
			if ( isset( $this->$prop ) ) {
				$this->$prop = clone $this->$prop;
			}
		}
		// Document fragments are special
		if ( isset( $this->caption ) ) {
			$oldCaption = $this->caption;
			$this->caption = $oldCaption->cloneNode( true );
			DOMDataUtils::dedupeNodeData( $oldCaption, $this->caption );
		}
		// Generic stdClass, use PHP serialization as a kludge
		foreach ( [ 'attrs', ] as $prop ) {
			if ( isset( $this->$prop ) ) {
				$this->$prop = unserialize( serialize( $this->$prop ) );
			}
		}
	}

	/** @inheritDoc */
	public static function jsonClassHintFor( string $keyname ) {
		static $hints = null;
		if ( $hints === null ) {
			$hints = [
				'attribs' => Hint::build( DataMwAttrib::class, Hint::USE_SQUARE, Hint::LIST ),
				// T367616: 'attrs' should be renamed to 'extAttribs' in
				// a future revision of the MediaWiki DOM Spec
				'attrs' => Hint::build( stdClass::class, Hint::ALLOW_OBJECT ),
				'body' => Hint::build( DataMwBody::class, Hint::ALLOW_OBJECT ),
				'wtOffsets' => Hint::build( SourceRange::class, Hint::USE_SQUARE ),
				'parts' => Hint::build( TemplateInfo::class, Hint::STDCLASS, Hint::LIST ),
				'errors' => Hint::build( DataMwError::class, Hint::LIST ),
				// 'caption' is not hinted as DocumentFragment because we
				// manually encode/decode it for MW Dom Spec 2.8.0 compat.
			];
		}
		return $hints[$keyname] ?? null;
	}

	/** @inheritDoc */
	public function toJsonArray( JsonCodecInterface $codec ): array {
		$result = (array)$this;
		// T367141: Third party clients (eg Cite) create arrays instead of
		// error objects.  We should convert them to proper DataMwError
		// objects once those exist.
		if ( isset( $result['errors'] ) ) {
			$result['errors'] = array_map(
				static fn ( $e ) => is_array( $e ) ? DataMwError::newFromJsonArray( $e ) :
					( $e instanceof DataMwError ? $e : DataMwError::newFromJsonArray( (array)$e ) ),
				$result['errors']
			);
		}
		// Legacy encoding of parts.
		if ( isset( $result['parts'] ) ) {
			$result['parts'] = array_map( static function ( $p ) {
				if ( $p instanceof TemplateInfo ) {
					$type = $p->type ?? 'template';
					if ( $type === 'parserfunction' ) {
						$type = 'template';
					} elseif ( $type === 'v3parserfunction' ) {
						$type = 'parserfunction';
					}
					$pp = (object)[];
					$pp->$type = $p;
					return $pp;
				}
				return $p;
			}, $result['parts'] );
		}
		// Caption compatibility with MediaWiki DOM Spec 2.8.0
		// See [[mw:Parsoid/MediaWiki DOM spec/Rich Attributes]] Phase 3
		// for discussion about alternate _h/_t marking for DocumentFragments
		if ( isset( $result['caption'] ) ) {
			$c = $codec->toJsonArray( $result['caption'], DocumentFragment::class );
			if ( is_string( $c['_h'] ?? null ) ) {
				$result['caption'] = $c['_h'];
			} else {
				$result['caption'] = $c;
			}
		}
		return $result;
	}

	/** @inheritDoc */
	public static function newFromJsonArray( JsonCodecInterface $codec, array $json ): DataMw {
		// Decode legacy encoding of parts.
		if ( isset( $json['parts'] ) ) {
			$json['parts'] = array_map( static function ( $p ) {
				if ( is_object( $p ) ) {
					$ptype = $type = 'template';
					if ( isset( $p->templatearg ) ) {
						$ptype = $type = 'templatearg';
					} elseif ( isset( $p->parserfunction ) ) {
						$type = 'parserfunction';
						$ptype = 'v3parserfunction';
					}
					$p = $p->$type;
					if ( isset( $p->func ) ) {
						$ptype = 'parserfunction';
					}
					$p->type = $ptype;
				}
				return $p;
			}, $json['parts'] );
		}
		// Usually '_h' or '_t' is used as a marker for captions, but
		// allow a bare string as well.
		$c = $json['caption'] ?? null;
		$c = is_string( $c ) ? [ '_h' => $c ] : $c;
		if ( $c !== null ) {
			$json['caption'] =
				$codec->newFromJsonArray( $c, DocumentFragment::class );
		}
		return new DataMw( $json );
	}

	/**
	 * Custom JsonClassCodec for DataMw.
	 *
	 * Because the 'caption' field has an embedded DocumentFragment that
	 * /doesn't/ use the standard encoding, we need to use a custom
	 * class codec which allows us to manually encode
	 * the DocumentFragment (by passing the codec itself to the
	 * serialization/deserialization methods).
	 */
	public static function jsonClassCodec(
		JsonCodecInterface $codec, ContainerInterface $serviceContainer
	): JsonClassCodec {
		return new class( $codec ) implements JsonClassCodec {
			private JsonCodecInterface $codec;

			public function __construct( JsonCodecInterface $codec ) {
				$this->codec = $codec;
			}

			/** @inheritDoc */
			public function toJsonArray( $obj ): array {
				return $obj->toJsonArray( $this->codec );
			}

			/** @inheritDoc */
			public function newFromJsonArray( string $className, array $json ) {
				return $className::newFromJsonArray( $this->codec, $json );
			}

			/** @inheritDoc */
			public function jsonClassHintFor( string $className, string $keyName ) {
				return $className::jsonClassHintFor( $keyName );
			}
		};
	}
}
