"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenIdAuthRoutes = void 0;
var _configSchema = require("@osd/config-schema");
var _cryptiles = require("@hapi/cryptiles");
var _querystring = require("querystring");
var _helper = require("./helper");
var _next_url = require("../../../utils/next_url");
var _common = require("../../../../common");
var _cookie_splitter = require("../../../session/cookie_splitter");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          *   Copyright OpenSearch Contributors
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          *   Licensed under the Apache License, Version 2.0 (the "License").
                                                                                                                                                                                                                                                                                                                          *   You may not use this file except in compliance with the License.
                                                                                                                                                                                                                                                                                                                          *   A copy of the License is located at
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          *       http://www.apache.org/licenses/LICENSE-2.0
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          *   or in the "license" file accompanying this file. This file is distributed
                                                                                                                                                                                                                                                                                                                          *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
                                                                                                                                                                                                                                                                                                                          *   express or implied. See the License for the specific language governing
                                                                                                                                                                                                                                                                                                                          *   permissions and limitations under the License.
                                                                                                                                                                                                                                                                                                                          */
class OpenIdAuthRoutes {
  constructor(router, config, sessionStorageFactory, openIdAuthConfig, securityClient, core, wreckClient) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.openIdAuthConfig = openIdAuthConfig;
    this.securityClient = securityClient;
    this.core = core;
    this.wreckClient = wreckClient;
  }
  redirectToLogin(request, response) {
    this.sessionStorageFactory.asScoped(request).clear();
    return response.redirected({
      headers: {
        location: `${this.core.http.basePath.serverBasePath}${_common.OPENID_AUTH_LOGIN}`
      }
    });
  }
  getExtraAuthStorageOptions(logger) {
    // If we're here, we will always have the openid configuration
    return {
      cookiePrefix: this.config.openid.extra_storage.cookie_prefix,
      additionalCookies: this.config.openid.extra_storage.additional_cookies,
      logger
    };
  }
  setupRoutes() {
    this.router.get({
      path: _common.OPENID_AUTH_LOGIN,
      validate: {
        query: _configSchema.schema.object({
          code: _configSchema.schema.maybe(_configSchema.schema.string()),
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: nexturl => {
              return (0, _next_url.validateNextUrl)(nexturl, this.core.http.basePath.serverBasePath);
            }
          })),
          redirectHash: _configSchema.schema.maybe(_configSchema.schema.string()),
          state: _configSchema.schema.maybe(_configSchema.schema.string()),
          refresh: _configSchema.schema.maybe(_configSchema.schema.string())
        }, {
          unknowns: 'allow'
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      var _this$config$openid2, _this$config$openid3, _cookie$oidc2;
      // implementation refers to https://github.com/hapijs/bell/blob/master/lib/oauth.js
      // Sign-in initialization
      if (!request.query.code) {
        var _this$config$openid;
        const nonce = (0, _cryptiles.randomString)(OpenIdAuthRoutes.NONCE_LENGTH);
        const query = {
          client_id: (_this$config$openid = this.config.openid) === null || _this$config$openid === void 0 ? void 0 : _this$config$openid.client_id,
          response_type: _common.AUTH_RESPONSE_TYPE,
          redirect_uri: `${(0, _helper.getBaseRedirectUrl)(this.config, this.core, request)}${_common.OPENID_AUTH_LOGIN}`,
          state: nonce,
          scope: this.openIdAuthConfig.scope
        };
        (0, _helper.includeAdditionalParameters)(query, context, this.config);
        const queryString = (0, _querystring.stringify)(query);
        const location = `${this.openIdAuthConfig.authorizationEndpoint}?${queryString}`;
        const cookie = {
          oidc: {
            state: nonce,
            nextUrl: (0, _helper.getNextUrl)(this.config, this.core, request),
            redirectHash: request.query.redirectHash === 'true'
          },
          authType: _common.AuthType.OPEN_ID
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location
          }
        });
      }

      // Authentication callback
      // validate state first
      let cookie;
      try {
        var _cookie$oidc;
        cookie = await this.sessionStorageFactory.asScoped(request).get();
        if (!cookie || !((_cookie$oidc = cookie.oidc) !== null && _cookie$oidc !== void 0 && _cookie$oidc.state) || cookie.oidc.state !== request.query.state) {
          return this.redirectToLogin(request, response);
        }
      } catch (error) {
        return this.redirectToLogin(request, response);
      }
      const nextUrl = cookie.oidc.nextUrl;
      const clientId = (_this$config$openid2 = this.config.openid) === null || _this$config$openid2 === void 0 ? void 0 : _this$config$openid2.client_id;
      const clientSecret = (_this$config$openid3 = this.config.openid) === null || _this$config$openid3 === void 0 ? void 0 : _this$config$openid3.client_secret;
      const redirectHash = ((_cookie$oidc2 = cookie.oidc) === null || _cookie$oidc2 === void 0 ? void 0 : _cookie$oidc2.redirectHash) || false;
      const query = {
        grant_type: _common.AUTH_GRANT_TYPE,
        code: request.query.code,
        redirect_uri: `${(0, _helper.getBaseRedirectUrl)(this.config, this.core, request)}${_common.OPENID_AUTH_LOGIN}`,
        client_id: clientId,
        client_secret: clientSecret
      };
      (0, _helper.includeAdditionalParameters)(query, context, this.config);
      try {
        var _this$config$openid4;
        const tokenResponse = await (0, _helper.callTokenEndpoint)(this.openIdAuthConfig.tokenEndpoint, query, this.wreckClient);
        const user = await this.securityClient.authenticateWithHeader(request, this.openIdAuthConfig.authHeaderName, `Bearer ${tokenResponse.idToken}`);

        // set to cookie
        const sessionStorage = {
          username: user.username,
          credentials: {
            authHeaderValueExtra: true,
            expiryTime: (0, _helper.getExpirationDate)(tokenResponse)
          },
          authType: _common.AuthType.OPEN_ID,
          expiryTime: Date.now() + this.config.session.ttl
        };
        if ((_this$config$openid4 = this.config.openid) !== null && _this$config$openid4 !== void 0 && _this$config$openid4.refresh_tokens && tokenResponse.refreshToken) {
          Object.assign(sessionStorage.credentials, {
            refresh_token: tokenResponse.refreshToken
          });
        }
        (0, _cookie_splitter.setExtraAuthStorage)(request, `Bearer ${tokenResponse.idToken}`, this.getExtraAuthStorageOptions(context.security_plugin.logger));
        this.sessionStorageFactory.asScoped(request).set(sessionStorage);
        if (redirectHash) {
          return response.redirected({
            headers: {
              location: `${this.core.http.basePath.serverBasePath}/auth/openid/redirectUrlFragment?nextUrl=${escape(nextUrl)}`
            }
          });
        } else {
          return response.redirected({
            headers: {
              location: nextUrl
            }
          });
        }
      } catch (error) {
        context.security_plugin.logger.error(`OpenId authentication failed: ${error}`);
        if (error.toString().toLowerCase().includes('authentication exception')) {
          return response.unauthorized();
        } else {
          return this.redirectToLogin(request, response);
        }
      }
    });
    this.router.get({
      path: _common.OPENID_AUTH_LOGOUT,
      validate: false
    }, async (context, request, response) => {
      var _cookie$credentials, _this$config$openid5;
      const cookie = await this.sessionStorageFactory.asScoped(request).get();
      let tokenFromExtraStorage = '';
      const extraAuthStorageOptions = this.getExtraAuthStorageOptions(context.security_plugin.logger);
      if (cookie !== null && cookie !== void 0 && (_cookie$credentials = cookie.credentials) !== null && _cookie$credentials !== void 0 && _cookie$credentials.authHeaderValueExtra) {
        tokenFromExtraStorage = (0, _cookie_splitter.getExtraAuthStorageValue)(request, extraAuthStorageOptions);
      }
      (0, _cookie_splitter.clearSplitCookies)(request, extraAuthStorageOptions);
      this.sessionStorageFactory.asScoped(request).clear();

      // authHeaderValue is the bearer header, e.g. "Bearer <auth_token>"
      const token = tokenFromExtraStorage.length ? tokenFromExtraStorage.split(' ')[1] : cookie === null || cookie === void 0 ? void 0 : cookie.credentials.authHeaderValue.split(' ')[1]; // get auth token
      const nextUrl = (0, _helper.getBaseRedirectUrl)(this.config, this.core, request);
      const logoutQueryParams = {
        post_logout_redirect_uri: `${nextUrl}`,
        id_token_hint: token
      };
      const endSessionUrl = (0, _helper.composeLogoutUrl)((_this$config$openid5 = this.config.openid) === null || _this$config$openid5 === void 0 ? void 0 : _this$config$openid5.logout_url, this.openIdAuthConfig.endSessionEndpoint, logoutQueryParams);
      return response.redirected({
        headers: {
          location: endSessionUrl
        }
      });
    });

    // captureUrlFragment is the first route that will be invoked in the SP initiated login.
    // This route will execute the captureUrlFragment.js script.
    this.core.http.resources.register({
      path: '/auth/openid/captureUrlFragment',
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: nexturl => {
              return (0, _next_url.validateNextUrl)(nexturl, this.core.http.basePath.serverBasePath);
            }
          }))
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      const serverBasePath = this.core.http.basePath.serverBasePath;
      return response.renderHtml({
        body: `
          <!DOCTYPE html>
          <title>OSD OIDC Capture</title>
          <link rel="icon" href="data:,">
          <script src="${serverBasePath}/auth/openid/captureUrlFragment.js"></script>
        `
      });
    });

    // This script will store the URL Hash in browser's local storage.
    this.core.http.resources.register({
      path: '/auth/openid/captureUrlFragment.js',
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      return response.renderJs({
        body: `let oidcHash=window.location.hash.toString();
                let redirectHash = false;
                if (oidcHash !== "") {
                  window.localStorage.removeItem('oidcHash');
                  window.localStorage.setItem('oidcHash', oidcHash);
                    redirectHash = true;
                }
                let params = new URLSearchParams(window.location.search);
                let nextUrl = params.get("nextUrl");
                finalUrl = "login?redirectHash=" + encodeURIComponent(redirectHash);
                if (!!nextUrl) {
                  finalUrl += "&nextUrl=" + encodeURIComponent(nextUrl);
                }
                window.location.replace(finalUrl);
              `
      });
    });

    //  Once the User is authenticated the browser will be redirected to '/auth/openid/redirectUrlFragment'
    //  route, which will execute the redirectUrlFragment.js.
    this.core.http.resources.register({
      path: '/auth/openid/redirectUrlFragment',
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.any()
        })
      },
      options: {
        authRequired: true
      }
    }, async (context, request, response) => {
      const serverBasePath = this.core.http.basePath.serverBasePath;
      return response.renderHtml({
        body: `
          <!DOCTYPE html>
          <title>OSD OpenID Success</title>
          <link rel="icon" href="data:,">
          <script src="${serverBasePath}/auth/openid/redirectUrlFragment.js"></script>
        `
      });
    });

    // This script will pop the Hash from local storage if it exists.
    // And forward the browser to the next url.
    this.core.http.resources.register({
      path: '/auth/openid/redirectUrlFragment.js',
      validate: false,
      options: {
        authRequired: true
      }
    }, async (context, request, response) => {
      return response.renderJs({
        body: `let oidcHash=window.localStorage.getItem('oidcHash');
                window.localStorage.removeItem('oidcHash');
                let params = new URLSearchParams(window.location.search);
                let nextUrl = params.get("nextUrl");
                finalUrl = nextUrl + oidcHash;
                window.location.replace(finalUrl);
              `
      });
    });
  }
}
exports.OpenIdAuthRoutes = OpenIdAuthRoutes;
_defineProperty(OpenIdAuthRoutes, "NONCE_LENGTH", 22);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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