﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/apigateway/APIGatewayRequest.h>
#include <aws/apigateway/APIGateway_EXPORTS.h>
#include <aws/apigateway/model/GatewayResponseType.h>
#include <aws/apigateway/model/PatchOperation.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace APIGateway {
namespace Model {

/**
 * <p>Updates a GatewayResponse of a specified response type on the given
 * RestApi.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/apigateway-2015-07-09/UpdateGatewayResponseRequest">AWS
 * API Reference</a></p>
 */
class UpdateGatewayResponseRequest : public APIGatewayRequest {
 public:
  AWS_APIGATEWAY_API UpdateGatewayResponseRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateGatewayResponse"; }

  AWS_APIGATEWAY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The string identifier of the associated RestApi.</p>
   */
  inline const Aws::String& GetRestApiId() const { return m_restApiId; }
  inline bool RestApiIdHasBeenSet() const { return m_restApiIdHasBeenSet; }
  template <typename RestApiIdT = Aws::String>
  void SetRestApiId(RestApiIdT&& value) {
    m_restApiIdHasBeenSet = true;
    m_restApiId = std::forward<RestApiIdT>(value);
  }
  template <typename RestApiIdT = Aws::String>
  UpdateGatewayResponseRequest& WithRestApiId(RestApiIdT&& value) {
    SetRestApiId(std::forward<RestApiIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The response type of the associated GatewayResponse.</p>
   */
  inline GatewayResponseType GetResponseType() const { return m_responseType; }
  inline bool ResponseTypeHasBeenSet() const { return m_responseTypeHasBeenSet; }
  inline void SetResponseType(GatewayResponseType value) {
    m_responseTypeHasBeenSet = true;
    m_responseType = value;
  }
  inline UpdateGatewayResponseRequest& WithResponseType(GatewayResponseType value) {
    SetResponseType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For more information about supported patch operations, see <a
   * href="https://docs.aws.amazon.com/apigateway/latest/api/patch-operations.html">Patch
   * Operations</a>.</p>
   */
  inline const Aws::Vector<PatchOperation>& GetPatchOperations() const { return m_patchOperations; }
  inline bool PatchOperationsHasBeenSet() const { return m_patchOperationsHasBeenSet; }
  template <typename PatchOperationsT = Aws::Vector<PatchOperation>>
  void SetPatchOperations(PatchOperationsT&& value) {
    m_patchOperationsHasBeenSet = true;
    m_patchOperations = std::forward<PatchOperationsT>(value);
  }
  template <typename PatchOperationsT = Aws::Vector<PatchOperation>>
  UpdateGatewayResponseRequest& WithPatchOperations(PatchOperationsT&& value) {
    SetPatchOperations(std::forward<PatchOperationsT>(value));
    return *this;
  }
  template <typename PatchOperationsT = PatchOperation>
  UpdateGatewayResponseRequest& AddPatchOperations(PatchOperationsT&& value) {
    m_patchOperationsHasBeenSet = true;
    m_patchOperations.emplace_back(std::forward<PatchOperationsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_restApiId;

  GatewayResponseType m_responseType{GatewayResponseType::NOT_SET};

  Aws::Vector<PatchOperation> m_patchOperations;
  bool m_restApiIdHasBeenSet = false;
  bool m_responseTypeHasBeenSet = false;
  bool m_patchOperationsHasBeenSet = false;
};

}  // namespace Model
}  // namespace APIGateway
}  // namespace Aws
