﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/billingconductor/BillingConductorRequest.h>
#include <aws/billingconductor/BillingConductor_EXPORTS.h>
#include <aws/billingconductor/model/ListAccountAssociationsFilter.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace BillingConductor {
namespace Model {

/**
 */
class ListAccountAssociationsRequest : public BillingConductorRequest {
 public:
  AWS_BILLINGCONDUCTOR_API ListAccountAssociationsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListAccountAssociations"; }

  AWS_BILLINGCONDUCTOR_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p> The preferred billing period to get account associations. </p>
   */
  inline const Aws::String& GetBillingPeriod() const { return m_billingPeriod; }
  inline bool BillingPeriodHasBeenSet() const { return m_billingPeriodHasBeenSet; }
  template <typename BillingPeriodT = Aws::String>
  void SetBillingPeriod(BillingPeriodT&& value) {
    m_billingPeriodHasBeenSet = true;
    m_billingPeriod = std::forward<BillingPeriodT>(value);
  }
  template <typename BillingPeriodT = Aws::String>
  ListAccountAssociationsRequest& WithBillingPeriod(BillingPeriodT&& value) {
    SetBillingPeriod(std::forward<BillingPeriodT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The filter on the account ID of the linked account, or any of the
   * following:</p> <p> <code>MONITORED</code>: linked accounts that are associated
   * to billing groups.</p> <p> <code>UNMONITORED</code>: linked accounts that aren't
   * associated to billing groups.</p> <p> <code>Billing Group Arn</code>: linked
   * accounts that are associated to the provided billing group Arn. </p>
   */
  inline const ListAccountAssociationsFilter& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = ListAccountAssociationsFilter>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = ListAccountAssociationsFilter>
  ListAccountAssociationsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The pagination token that's used on subsequent calls to retrieve accounts.
   * </p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListAccountAssociationsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_billingPeriod;

  ListAccountAssociationsFilter m_filters;

  Aws::String m_nextToken;
  bool m_billingPeriodHasBeenSet = false;
  bool m_filtersHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace BillingConductor
}  // namespace Aws
