﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/dms/DatabaseMigrationServiceRequest.h>
#include <aws/dms/DatabaseMigrationService_EXPORTS.h>
#include <aws/dms/model/KerberosAuthenticationSettings.h>

#include <utility>

namespace Aws {
namespace DatabaseMigrationService {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/dms-2016-01-01/ModifyReplicationInstanceMessage">AWS
 * API Reference</a></p>
 */
class ModifyReplicationInstanceRequest : public DatabaseMigrationServiceRequest {
 public:
  AWS_DATABASEMIGRATIONSERVICE_API ModifyReplicationInstanceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ModifyReplicationInstance"; }

  AWS_DATABASEMIGRATIONSERVICE_API Aws::String SerializePayload() const override;

  AWS_DATABASEMIGRATIONSERVICE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the replication instance.</p>
   */
  inline const Aws::String& GetReplicationInstanceArn() const { return m_replicationInstanceArn; }
  inline bool ReplicationInstanceArnHasBeenSet() const { return m_replicationInstanceArnHasBeenSet; }
  template <typename ReplicationInstanceArnT = Aws::String>
  void SetReplicationInstanceArn(ReplicationInstanceArnT&& value) {
    m_replicationInstanceArnHasBeenSet = true;
    m_replicationInstanceArn = std::forward<ReplicationInstanceArnT>(value);
  }
  template <typename ReplicationInstanceArnT = Aws::String>
  ModifyReplicationInstanceRequest& WithReplicationInstanceArn(ReplicationInstanceArnT&& value) {
    SetReplicationInstanceArn(std::forward<ReplicationInstanceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The amount of storage (in gigabytes) to be allocated for the replication
   * instance.</p>
   */
  inline int GetAllocatedStorage() const { return m_allocatedStorage; }
  inline bool AllocatedStorageHasBeenSet() const { return m_allocatedStorageHasBeenSet; }
  inline void SetAllocatedStorage(int value) {
    m_allocatedStorageHasBeenSet = true;
    m_allocatedStorage = value;
  }
  inline ModifyReplicationInstanceRequest& WithAllocatedStorage(int value) {
    SetAllocatedStorage(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether the changes should be applied immediately or during the
   * next maintenance window.</p>
   */
  inline bool GetApplyImmediately() const { return m_applyImmediately; }
  inline bool ApplyImmediatelyHasBeenSet() const { return m_applyImmediatelyHasBeenSet; }
  inline void SetApplyImmediately(bool value) {
    m_applyImmediatelyHasBeenSet = true;
    m_applyImmediately = value;
  }
  inline ModifyReplicationInstanceRequest& WithApplyImmediately(bool value) {
    SetApplyImmediately(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The compute and memory capacity of the replication instance as defined for
   * the specified replication instance class. For example to specify the instance
   * class dms.c4.large, set this parameter to <code>"dms.c4.large"</code>.</p>
   * <p>For more information on the settings and capacities for the available
   * replication instance classes, see <a
   * href="https://docs.aws.amazon.com/dms/latest/userguide/CHAP_ReplicationInstance.html#CHAP_ReplicationInstance.InDepth">
   * Selecting the right DMS replication instance for your migration</a>. </p>
   */
  inline const Aws::String& GetReplicationInstanceClass() const { return m_replicationInstanceClass; }
  inline bool ReplicationInstanceClassHasBeenSet() const { return m_replicationInstanceClassHasBeenSet; }
  template <typename ReplicationInstanceClassT = Aws::String>
  void SetReplicationInstanceClass(ReplicationInstanceClassT&& value) {
    m_replicationInstanceClassHasBeenSet = true;
    m_replicationInstanceClass = std::forward<ReplicationInstanceClassT>(value);
  }
  template <typename ReplicationInstanceClassT = Aws::String>
  ModifyReplicationInstanceRequest& WithReplicationInstanceClass(ReplicationInstanceClassT&& value) {
    SetReplicationInstanceClass(std::forward<ReplicationInstanceClassT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Specifies the VPC security group to be used with the replication instance.
   * The VPC security group must work with the VPC containing the replication
   * instance. </p>
   */
  inline const Aws::Vector<Aws::String>& GetVpcSecurityGroupIds() const { return m_vpcSecurityGroupIds; }
  inline bool VpcSecurityGroupIdsHasBeenSet() const { return m_vpcSecurityGroupIdsHasBeenSet; }
  template <typename VpcSecurityGroupIdsT = Aws::Vector<Aws::String>>
  void SetVpcSecurityGroupIds(VpcSecurityGroupIdsT&& value) {
    m_vpcSecurityGroupIdsHasBeenSet = true;
    m_vpcSecurityGroupIds = std::forward<VpcSecurityGroupIdsT>(value);
  }
  template <typename VpcSecurityGroupIdsT = Aws::Vector<Aws::String>>
  ModifyReplicationInstanceRequest& WithVpcSecurityGroupIds(VpcSecurityGroupIdsT&& value) {
    SetVpcSecurityGroupIds(std::forward<VpcSecurityGroupIdsT>(value));
    return *this;
  }
  template <typename VpcSecurityGroupIdsT = Aws::String>
  ModifyReplicationInstanceRequest& AddVpcSecurityGroupIds(VpcSecurityGroupIdsT&& value) {
    m_vpcSecurityGroupIdsHasBeenSet = true;
    m_vpcSecurityGroupIds.emplace_back(std::forward<VpcSecurityGroupIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The weekly time range (in UTC) during which system maintenance can occur,
   * which might result in an outage. Changing this parameter does not result in an
   * outage, except in the following situation, and the change is asynchronously
   * applied as soon as possible. If moving this window to the current time, there
   * must be at least 30 minutes between the current time and end of the window to
   * ensure pending changes are applied.</p> <p>Default: Uses existing setting</p>
   * <p>Format: ddd:hh24:mi-ddd:hh24:mi</p> <p>Valid Days: Mon | Tue | Wed | Thu |
   * Fri | Sat | Sun</p> <p>Constraints: Must be at least 30 minutes</p>
   */
  inline const Aws::String& GetPreferredMaintenanceWindow() const { return m_preferredMaintenanceWindow; }
  inline bool PreferredMaintenanceWindowHasBeenSet() const { return m_preferredMaintenanceWindowHasBeenSet; }
  template <typename PreferredMaintenanceWindowT = Aws::String>
  void SetPreferredMaintenanceWindow(PreferredMaintenanceWindowT&& value) {
    m_preferredMaintenanceWindowHasBeenSet = true;
    m_preferredMaintenanceWindow = std::forward<PreferredMaintenanceWindowT>(value);
  }
  template <typename PreferredMaintenanceWindowT = Aws::String>
  ModifyReplicationInstanceRequest& WithPreferredMaintenanceWindow(PreferredMaintenanceWindowT&& value) {
    SetPreferredMaintenanceWindow(std::forward<PreferredMaintenanceWindowT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Specifies whether the replication instance is a Multi-AZ deployment. You
   * can't set the <code>AvailabilityZone</code> parameter if the Multi-AZ parameter
   * is set to <code>true</code>. </p>
   */
  inline bool GetMultiAZ() const { return m_multiAZ; }
  inline bool MultiAZHasBeenSet() const { return m_multiAZHasBeenSet; }
  inline void SetMultiAZ(bool value) {
    m_multiAZHasBeenSet = true;
    m_multiAZ = value;
  }
  inline ModifyReplicationInstanceRequest& WithMultiAZ(bool value) {
    SetMultiAZ(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The engine version number of the replication instance.</p> <p>When modifying
   * a major engine version of an instance, also set
   * <code>AllowMajorVersionUpgrade</code> to <code>true</code>.</p>
   */
  inline const Aws::String& GetEngineVersion() const { return m_engineVersion; }
  inline bool EngineVersionHasBeenSet() const { return m_engineVersionHasBeenSet; }
  template <typename EngineVersionT = Aws::String>
  void SetEngineVersion(EngineVersionT&& value) {
    m_engineVersionHasBeenSet = true;
    m_engineVersion = std::forward<EngineVersionT>(value);
  }
  template <typename EngineVersionT = Aws::String>
  ModifyReplicationInstanceRequest& WithEngineVersion(EngineVersionT&& value) {
    SetEngineVersion(std::forward<EngineVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates that major version upgrades are allowed. Changing this parameter
   * does not result in an outage, and the change is asynchronously applied as soon
   * as possible.</p> <p>This parameter must be set to <code>true</code> when
   * specifying a value for the <code>EngineVersion</code> parameter that is a
   * different major version than the replication instance's current version.</p>
   */
  inline bool GetAllowMajorVersionUpgrade() const { return m_allowMajorVersionUpgrade; }
  inline bool AllowMajorVersionUpgradeHasBeenSet() const { return m_allowMajorVersionUpgradeHasBeenSet; }
  inline void SetAllowMajorVersionUpgrade(bool value) {
    m_allowMajorVersionUpgradeHasBeenSet = true;
    m_allowMajorVersionUpgrade = value;
  }
  inline ModifyReplicationInstanceRequest& WithAllowMajorVersionUpgrade(bool value) {
    SetAllowMajorVersionUpgrade(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value that indicates that minor version upgrades are applied automatically
   * to the replication instance during the maintenance window. Changing this
   * parameter doesn't result in an outage, except in the case described following.
   * The change is asynchronously applied as soon as possible. </p> <p>An outage does
   * result if these factors apply: </p> <ul> <li> <p>This parameter is set to
   * <code>true</code> during the maintenance window.</p> </li> <li> <p>A newer minor
   * version is available. </p> </li> <li> <p>DMS has enabled automatic patching for
   * the given engine version. </p> </li> </ul>
   */
  inline bool GetAutoMinorVersionUpgrade() const { return m_autoMinorVersionUpgrade; }
  inline bool AutoMinorVersionUpgradeHasBeenSet() const { return m_autoMinorVersionUpgradeHasBeenSet; }
  inline void SetAutoMinorVersionUpgrade(bool value) {
    m_autoMinorVersionUpgradeHasBeenSet = true;
    m_autoMinorVersionUpgrade = value;
  }
  inline ModifyReplicationInstanceRequest& WithAutoMinorVersionUpgrade(bool value) {
    SetAutoMinorVersionUpgrade(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The replication instance identifier. This parameter is stored as a lowercase
   * string.</p>
   */
  inline const Aws::String& GetReplicationInstanceIdentifier() const { return m_replicationInstanceIdentifier; }
  inline bool ReplicationInstanceIdentifierHasBeenSet() const { return m_replicationInstanceIdentifierHasBeenSet; }
  template <typename ReplicationInstanceIdentifierT = Aws::String>
  void SetReplicationInstanceIdentifier(ReplicationInstanceIdentifierT&& value) {
    m_replicationInstanceIdentifierHasBeenSet = true;
    m_replicationInstanceIdentifier = std::forward<ReplicationInstanceIdentifierT>(value);
  }
  template <typename ReplicationInstanceIdentifierT = Aws::String>
  ModifyReplicationInstanceRequest& WithReplicationInstanceIdentifier(ReplicationInstanceIdentifierT&& value) {
    SetReplicationInstanceIdentifier(std::forward<ReplicationInstanceIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of IP address protocol used by a replication instance, such as IPv4
   * only or Dual-stack that supports both IPv4 and IPv6 addressing. IPv6 only is not
   * yet supported.</p>
   */
  inline const Aws::String& GetNetworkType() const { return m_networkType; }
  inline bool NetworkTypeHasBeenSet() const { return m_networkTypeHasBeenSet; }
  template <typename NetworkTypeT = Aws::String>
  void SetNetworkType(NetworkTypeT&& value) {
    m_networkTypeHasBeenSet = true;
    m_networkType = std::forward<NetworkTypeT>(value);
  }
  template <typename NetworkTypeT = Aws::String>
  ModifyReplicationInstanceRequest& WithNetworkType(NetworkTypeT&& value) {
    SetNetworkType(std::forward<NetworkTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the settings required for kerberos authentication when modifying a
   * replication instance.</p>
   */
  inline const KerberosAuthenticationSettings& GetKerberosAuthenticationSettings() const { return m_kerberosAuthenticationSettings; }
  inline bool KerberosAuthenticationSettingsHasBeenSet() const { return m_kerberosAuthenticationSettingsHasBeenSet; }
  template <typename KerberosAuthenticationSettingsT = KerberosAuthenticationSettings>
  void SetKerberosAuthenticationSettings(KerberosAuthenticationSettingsT&& value) {
    m_kerberosAuthenticationSettingsHasBeenSet = true;
    m_kerberosAuthenticationSettings = std::forward<KerberosAuthenticationSettingsT>(value);
  }
  template <typename KerberosAuthenticationSettingsT = KerberosAuthenticationSettings>
  ModifyReplicationInstanceRequest& WithKerberosAuthenticationSettings(KerberosAuthenticationSettingsT&& value) {
    SetKerberosAuthenticationSettings(std::forward<KerberosAuthenticationSettingsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_replicationInstanceArn;

  int m_allocatedStorage{0};

  bool m_applyImmediately{false};

  Aws::String m_replicationInstanceClass;

  Aws::Vector<Aws::String> m_vpcSecurityGroupIds;

  Aws::String m_preferredMaintenanceWindow;

  bool m_multiAZ{false};

  Aws::String m_engineVersion;

  bool m_allowMajorVersionUpgrade{false};

  bool m_autoMinorVersionUpgrade{false};

  Aws::String m_replicationInstanceIdentifier;

  Aws::String m_networkType;

  KerberosAuthenticationSettings m_kerberosAuthenticationSettings;
  bool m_replicationInstanceArnHasBeenSet = false;
  bool m_allocatedStorageHasBeenSet = false;
  bool m_applyImmediatelyHasBeenSet = false;
  bool m_replicationInstanceClassHasBeenSet = false;
  bool m_vpcSecurityGroupIdsHasBeenSet = false;
  bool m_preferredMaintenanceWindowHasBeenSet = false;
  bool m_multiAZHasBeenSet = false;
  bool m_engineVersionHasBeenSet = false;
  bool m_allowMajorVersionUpgradeHasBeenSet = false;
  bool m_autoMinorVersionUpgradeHasBeenSet = false;
  bool m_replicationInstanceIdentifierHasBeenSet = false;
  bool m_networkTypeHasBeenSet = false;
  bool m_kerberosAuthenticationSettingsHasBeenSet = false;
};

}  // namespace Model
}  // namespace DatabaseMigrationService
}  // namespace Aws
