﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/inspector2/Inspector2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Inspector2 {
namespace Model {

/**
 * <p>The Common Vulnerability Scoring System (CVSS) version 4 details for the
 * vulnerability.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/inspector2-2020-06-08/Cvss4">AWS
 * API Reference</a></p>
 */
class Cvss4 {
 public:
  AWS_INSPECTOR2_API Cvss4() = default;
  AWS_INSPECTOR2_API Cvss4(Aws::Utils::Json::JsonView jsonValue);
  AWS_INSPECTOR2_API Cvss4& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_INSPECTOR2_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The base CVSS v4 score for the vulnerability finding, which rates the
   * severity of the vulnerability on a scale from 0 to 10.</p>
   */
  inline double GetBaseScore() const { return m_baseScore; }
  inline bool BaseScoreHasBeenSet() const { return m_baseScoreHasBeenSet; }
  inline void SetBaseScore(double value) {
    m_baseScoreHasBeenSet = true;
    m_baseScore = value;
  }
  inline Cvss4& WithBaseScore(double value) {
    SetBaseScore(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The CVSS v4 scoring vector, which contains the metrics and measurements that
   * were used to calculate the base score.</p>
   */
  inline const Aws::String& GetScoringVector() const { return m_scoringVector; }
  inline bool ScoringVectorHasBeenSet() const { return m_scoringVectorHasBeenSet; }
  template <typename ScoringVectorT = Aws::String>
  void SetScoringVector(ScoringVectorT&& value) {
    m_scoringVectorHasBeenSet = true;
    m_scoringVector = std::forward<ScoringVectorT>(value);
  }
  template <typename ScoringVectorT = Aws::String>
  Cvss4& WithScoringVector(ScoringVectorT&& value) {
    SetScoringVector(std::forward<ScoringVectorT>(value));
    return *this;
  }
  ///@}
 private:
  double m_baseScore{0.0};

  Aws::String m_scoringVector;
  bool m_baseScoreHasBeenSet = false;
  bool m_scoringVectorHasBeenSet = false;
};

}  // namespace Model
}  // namespace Inspector2
}  // namespace Aws
