﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/xray/XRay_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace XRay {
namespace Model {

/**
 * <p>A resource policy grants one or more Amazon Web Services services and
 * accounts permissions to access X-Ray. Each resource policy is associated with a
 * specific Amazon Web Services account.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/ResourcePolicy">AWS
 * API Reference</a></p>
 */
class ResourcePolicy {
 public:
  AWS_XRAY_API ResourcePolicy() = default;
  AWS_XRAY_API ResourcePolicy(Aws::Utils::Json::JsonView jsonValue);
  AWS_XRAY_API ResourcePolicy& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_XRAY_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the resource policy. Must be unique within a specific Amazon Web
   * Services account.</p>
   */
  inline const Aws::String& GetPolicyName() const { return m_policyName; }
  inline bool PolicyNameHasBeenSet() const { return m_policyNameHasBeenSet; }
  template <typename PolicyNameT = Aws::String>
  void SetPolicyName(PolicyNameT&& value) {
    m_policyNameHasBeenSet = true;
    m_policyName = std::forward<PolicyNameT>(value);
  }
  template <typename PolicyNameT = Aws::String>
  ResourcePolicy& WithPolicyName(PolicyNameT&& value) {
    SetPolicyName(std::forward<PolicyNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The resource policy document, which can be up to 5kb in size.</p>
   */
  inline const Aws::String& GetPolicyDocument() const { return m_policyDocument; }
  inline bool PolicyDocumentHasBeenSet() const { return m_policyDocumentHasBeenSet; }
  template <typename PolicyDocumentT = Aws::String>
  void SetPolicyDocument(PolicyDocumentT&& value) {
    m_policyDocumentHasBeenSet = true;
    m_policyDocument = std::forward<PolicyDocumentT>(value);
  }
  template <typename PolicyDocumentT = Aws::String>
  ResourcePolicy& WithPolicyDocument(PolicyDocumentT&& value) {
    SetPolicyDocument(std::forward<PolicyDocumentT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Returns the current policy revision id for this policy name.</p>
   */
  inline const Aws::String& GetPolicyRevisionId() const { return m_policyRevisionId; }
  inline bool PolicyRevisionIdHasBeenSet() const { return m_policyRevisionIdHasBeenSet; }
  template <typename PolicyRevisionIdT = Aws::String>
  void SetPolicyRevisionId(PolicyRevisionIdT&& value) {
    m_policyRevisionIdHasBeenSet = true;
    m_policyRevisionId = std::forward<PolicyRevisionIdT>(value);
  }
  template <typename PolicyRevisionIdT = Aws::String>
  ResourcePolicy& WithPolicyRevisionId(PolicyRevisionIdT&& value) {
    SetPolicyRevisionId(std::forward<PolicyRevisionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When the policy was last updated, in Unix time seconds.</p>
   */
  inline const Aws::Utils::DateTime& GetLastUpdatedTime() const { return m_lastUpdatedTime; }
  inline bool LastUpdatedTimeHasBeenSet() const { return m_lastUpdatedTimeHasBeenSet; }
  template <typename LastUpdatedTimeT = Aws::Utils::DateTime>
  void SetLastUpdatedTime(LastUpdatedTimeT&& value) {
    m_lastUpdatedTimeHasBeenSet = true;
    m_lastUpdatedTime = std::forward<LastUpdatedTimeT>(value);
  }
  template <typename LastUpdatedTimeT = Aws::Utils::DateTime>
  ResourcePolicy& WithLastUpdatedTime(LastUpdatedTimeT&& value) {
    SetLastUpdatedTime(std::forward<LastUpdatedTimeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_policyName;

  Aws::String m_policyDocument;

  Aws::String m_policyRevisionId;

  Aws::Utils::DateTime m_lastUpdatedTime{};
  bool m_policyNameHasBeenSet = false;
  bool m_policyDocumentHasBeenSet = false;
  bool m_policyRevisionIdHasBeenSet = false;
  bool m_lastUpdatedTimeHasBeenSet = false;
};

}  // namespace Model
}  // namespace XRay
}  // namespace Aws
