/*
 * Copyright (c) 2004-2005 The Trustees of Indiana University and Indiana
 *                         University Research and Technology
 *                         Corporation.  All rights reserved.
 * Copyright (c) 2004-2005 The University of Tennessee and The University
 *                         of Tennessee Research Foundation.  All rights
 *                         reserved.
 * Copyright (c) 2004-2010 High Performance Computing Center Stuttgart, 
 *                         University of Stuttgart.  All rights reserved.
 * Copyright (c) 2004-2005 The Regents of the University of California.
 *                         All rights reserved.
 * Copyright (c) 2009      Sun Microsystems, Inc.  All rights reserved.
 * Copyright (c) 2009-2013 Cisco Systems, Inc.  All rights reserved.
 * Copyright (c) 2011-2013 UT-Battelle, LLC. All rights reserved.
 * Copyright (C) 2013      Mellanox Technologies Ltd. All rights reserved.
 * $COPYRIGHT$
 * 
 * Additional copyrights may follow
 * 
 * $HEADER$
 *
 * This file is included at the bottom of ocoms_config.h, and is
 * therefore a) after all the #define's that were output from
 * configure, and b) included in most/all files in Open MPI.
 *
 * Since this file is *only* ever included by ocoms_config.h, and
 * ocoms_config.h already has #ifndef/#endif protection, there is no
 * need to #ifndef/#endif protection here.
 */

#ifndef OCOMS_CONFIG_H 
#error "ocoms_config_bottom.h should only be included from ocoms_config.h"
#endif

/*
 * If we build a static library, Visual C define the _LIB symbol. In the
 * case of a shared library _USERDLL get defined.
 *
 * OMPI_BUILDING and _LIB define how ompi_config.h
 * handles configuring all of Open MPI's "compatibility" code.  Both
 * constants will always be defined by the end of ompi_config.h.
 *
 * OMPI_BUILDING affects how much compatibility code is included by
 * ompi_config.h.  It will always be 1 or 0.  The user can set the
 * value before including either mpi.h or ompi_config.h and it will be
 * respected.  If ompi_config.h is included before mpi.h, it will
 * default to 1.  If mpi.h is included before ompi_config.h, it will
 * default to 0.
 */
#ifndef OMPI_BUILDING
#define OMPI_BUILDING 1
#endif

/*
 * Flex is trying to include the unistd.h file. As there is no configure
 * option or this, the flex generated files will try to include the file
 * even on platforms without unistd.h. Therefore, if we
 * know this file is not available, we can prevent flex from including it.
 */
#ifndef HAVE_UNISTD_H
#define YY_NO_UNISTD_H
#endif

/***********************************************************************
 *
 * code that should be in ompi_config_bottom.h regardless of build
 * status
 *
 **********************************************************************/

/* Do we have posix or solaris thread lib */
#define OCOMS_HAVE_THREADS (OCOMS_HAVE_POSIX_THREADS || OCOMS_HAVE_SOLARIS_THREADS)

/*
 * BEGIN_C_DECLS should be used at the beginning of your declarations,
 * so that C++ compilers don't mangle their names.  Use END_C_DECLS at
 * the end of C declarations.
 */
#undef BEGIN_C_DECLS
#undef END_C_DECLS
#if defined(c_plusplus) || defined(__cplusplus)
# define BEGIN_C_DECLS extern "C" {
# define END_C_DECLS }
#else
#define BEGIN_C_DECLS          /* empty */
#define END_C_DECLS            /* empty */
#endif

/**
 * The attribute definition should be included before any potential
 * usage.
 */
#if OCOMS_HAVE_ATTRIBUTE_ALIGNED
#    define __ocoms_attribute_aligned__(a)    __attribute__((__aligned__(a)))
#    define __ocoms_attribute_aligned_max__   __attribute__((__aligned__))
#else
#    define __ocoms_attribute_aligned__(a)
#    define __ocoms_attribute_aligned_max__
#endif

#if OCOMS_HAVE_ATTRIBUTE_ALWAYS_INLINE
#    define __ocoms_attribute_always_inline__ __attribute__((__always_inline__))
#else
#    define __ocoms_attribute_always_inline__
#endif

#if OCOMS_HAVE_ATTRIBUTE_COLD
#    define __ocoms_attribute_cold__          __attribute__((__cold__))
#else
#    define __ocoms_attribute_cold__
#endif

#if OCOMS_HAVE_ATTRIBUTE_CONST
#    define __ocoms_attribute_const__         __attribute__((__const__))
#else
#    define __ocoms_attribute_const__
#endif

#if OCOMS_HAVE_ATTRIBUTE_DEPRECATED
#    define __ocoms_attribute_deprecated__    __attribute__((__deprecated__))
#else
#    define __ocoms_attribute_deprecated__
#endif

#if OCOMS_HAVE_ATTRIBUTE_FORMAT
#    define __ocoms_attribute_format__(a,b,c) __attribute__((__format__(a, b, c)))
#else
#    define __ocoms_attribute_format__(a,b,c)
#endif

/* Use this __atribute__ on function-ptr declarations, only */
#if OCOMS_HAVE_ATTRIBUTE_FORMAT_FUNCPTR
#    define __ocoms_attribute_format_funcptr__(a,b,c) __attribute__((__format__(a, b, c)))
#else
#    define __ocoms_attribute_format_funcptr__(a,b,c)
#endif

#if OCOMS_HAVE_ATTRIBUTE_HOT
#    define __ocoms_attribute_hot__           __attribute__((__hot__))
#else
#    define __ocoms_attribute_hot__
#endif

#if OCOMS_HAVE_ATTRIBUTE_MALLOC
#    define __ocoms_attribute_malloc__        __attribute__((__malloc__))
#else
#    define __ocoms_attribute_malloc__
#endif

#if OCOMS_HAVE_ATTRIBUTE_MAY_ALIAS
#    define __ocoms_attribute_may_alias__     __attribute__((__may_alias__))
#else
#    define __ocoms_attribute_may_alias__
#endif

#if OCOMS_HAVE_ATTRIBUTE_NO_INSTRUMENT_FUNCTION
#    define __ocoms_attribute_no_instrument_function__  __attribute__((__no_instrument_function__))
#else
#    define __ocoms_attribute_no_instrument_function__
#endif

#if OCOMS_HAVE_ATTRIBUTE_NOINLINE
#    define __ocoms_attribute_noinline__  __attribute__((__noinline__))
#else
#    define __ocoms_attribute_noinline__
#endif

#if OCOMS_HAVE_ATTRIBUTE_NONNULL
#    define __ocoms_attribute_nonnull__(a)    __attribute__((__nonnull__(a)))
#    define __ocoms_attribute_nonnull_all__   __attribute__((__nonnull__))
#else
#    define __ocoms_attribute_nonnull__(a)
#    define __ocoms_attribute_nonnull_all__
#endif

#if OCOMS_HAVE_ATTRIBUTE_NORETURN
#    define __ocoms_attribute_noreturn__      __attribute__((__noreturn__))
#else
#    define __ocoms_attribute_noreturn__
#endif

/* Use this __atribute__ on function-ptr declarations, only */
#if OCOMS_HAVE_ATTRIBUTE_NORETURN_FUNCPTR
#    define __ocoms_attribute_noreturn_funcptr__  __attribute__((__noreturn__))
#else
#    define __ocoms_attribute_noreturn_funcptr__
#endif

#if OCOMS_HAVE_ATTRIBUTE_PACKED
#    define __ocoms_attribute_packed__        __attribute__((__packed__))
#else
#    define __ocoms_attribute_packed__
#endif

#if OCOMS_HAVE_ATTRIBUTE_PURE
#    define __ocoms_attribute_pure__          __attribute__((__pure__))
#else
#    define __ocoms_attribute_pure__
#endif

#if OCOMS_HAVE_ATTRIBUTE_SENTINEL
#    define __ocoms_attribute_sentinel__      __attribute__((__sentinel__))
#else
#    define __ocoms_attribute_sentinel__
#endif

#if OCOMS_HAVE_ATTRIBUTE_UNUSED
#    define __ocoms_attribute_unused__        __attribute__((__unused__))
#else
#    define __ocoms_attribute_unused__
#endif

#if OCOMS_HAVE_ATTRIBUTE_VISIBILITY
#    define __ocoms_attribute_visibility__(a) __attribute__((__visibility__(a)))
#else
#    define __ocoms_attribute_visibility__(a)
#endif

#if OCOMS_HAVE_ATTRIBUTE_WARN_UNUSED_RESULT
#    define __ocoms_attribute_warn_unused_result__ __attribute__((__warn_unused_result__))
#else
#    define __ocoms_attribute_warn_unused_result__
#endif

#if OCOMS_HAVE_ATTRIBUTE_WEAK_ALIAS
#    define __ocoms_attribute_weak_alias__(a) __attribute__((__weak__, __alias__(a)))
#else
#    define __ocoms_attribute_weak_alias__(a)
#endif

#if OCOMS_HAVE_ATTRIBUTE_DESTRUCTOR
#    define __ocoms_attribute_destructor__    __attribute__((__destructor__))
#else
#    define __ocoms_attribute_destructor__
#endif

#  if OCOMS_C_HAVE_VISIBILITY
#    define OCOMS_DECLSPEC           __ocoms_attribute_visibility__("default")
#    define OCOMS_MODULE_DECLSPEC    __ocoms_attribute_visibility__("default")
#  else
#    define OCOMS_DECLSPEC
#    define OCOMS_MODULE_DECLSPEC
#  endif

/*
 * Do we have <stdint.h>?
 */
#ifdef HAVE_STDINT_H
#if !defined(__STDC_LIMIT_MACROS) && (defined(c_plusplus) || defined (__cplusplus))
/* When using a C++ compiler, the max / min value #defines for std
   types are only included if __STDC_LIMIT_MACROS is set before
   including stdint.h */
#define __STDC_LIMIT_MACROS
#endif
#include "ocoms/platform/ocoms_config.h"
#include <stdint.h>
#else
#include "ocoms_stdint.h"
#endif

/***********************************************************************
 *
 * Code that is only for when building Open MPI or utilities that are
 * using the internals of Open MPI.  It should not be included when
 * building MPI applications
 *
 **********************************************************************/
#if OMPI_BUILDING

#ifndef HAVE_PTRDIFF_T
typedef OCOMS_PTRDIFF_TYPE ptrdiff_t;
#endif

/*
 * If we're in C, we may need to bring in the bool type and true/false
 * constants.  OCOMS_NEED_C_BOOL will be true if the compiler either
 * needs <stdbool.h> or doesn't define the bool type at all.
 */
#if !(defined(c_plusplus) || defined(__cplusplus))
#    if OCOMS_NEED_C_BOOL
#        if OCOMS_USE_STDBOOL_H
             /* If we're using <stdbool.h>, there is an implicit
                assumption that the C++ bool is the same size and has
                the same alignment.  However, configure may have
                disabled the MPI C++ bindings, so if "_Bool" exists,
                then use that sizeof. */
#            include <stdbool.h>
             /* This section exists because AC_SIZEOF(bool) may not be
                run in configure if we're not building the MPI C++
                bindings. */
#            undef SIZEOF_BOOL
#            if SIZEOF__BOOL > 0
#                define SIZEOF_BOOL SIZEOF__BOOL
#            else
                 /* If all else fails, assume it's 1 */
#                define SIZEOF_BOOL 1
#            endif
#        else
             /* We need to create a bool type and ensure that it's the
                same size / alignment as the C++ bool size /
                alignment */
#            define false 0
#            define true 1
#            if SIZEOF_BOOL == SIZEOF_CHAR && OCOMS_ALIGNMENT_CXX_BOOL == OCOMS_ALIGNMENT_CHAR
typedef unsigned char bool;
#            elif SIZEOF_BOOL == SIZEOF_SHORT && OCOMS_ALIGNMENT_CXX_BOOL == OCOMS_ALIGNMENT_SHORT
typedef short bool;
#            elif SIZEOF_BOOL == SIZEOF_INT && OCOMS_ALIGNMENT_CXX_BOOL == OCOMS_ALIGNMENT_INT
typedef int bool;
#            elif SIZEOF_BOOL == SIZEOF_LONG && OCOMS_ALIGNMENT_CXX_BOOL == OCOMS_ALIGNMENT_LONG
typedef long bool;
#            elif defined(SIZEOF_LONG_LONG) && defined(OCOMS_ALIGNMENT_LONG) && SIZEOF_BOOL == SIZEOF_LONG && OCOMS_ALIGNMENT_CXX_BOOL == OCOMS_ALIGNMENT_LONG
typedef long long bool;
             /* If we have _Bool, use that */
#            elif SIZEOF__BOOL > 0
#                undef SIZEOF_BOOL
#                define bool _Bool
#                define SIZEOF_BOOL SIZEOF__BOOL
#            else
             /* If all else fails, just make bool be an unsigned char
                and size of 1 */
typedef unsigned char bool;
#                define SIZEOF_BOOL 1
#            endif
#        endif  /* OCOMS_USE_STDBOOL_H */
#    endif  /* OCOMS_NEED_C_BOOL */
#endif

/*
 * Maximum size of a filename path.
 */
#include <limits.h>
#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif
#if defined(PATH_MAX)
#define OCOMS_PATH_MAX	(PATH_MAX + 1)
#elif defined(_POSIX_PATH_MAX)
#define OCOMS_PATH_MAX	(_POSIX_PATH_MAX + 1)
#else
#define OCOMS_PATH_MAX	256
#endif

/*
 * Set the compile-time path-separator on this system and variable separator
 */
#define OCOMS_PATH_SEP "/"
#define OCOMS_ENV_SEP  ':'


/*
 * Do we want memory debugging?
 *
 * A few scenarios:
 *
 * 1. In the OMPI C library: we want these defines in all cases
 * 2. In the OMPI C++ bindings: we do not want them
 * 3. In the OMPI C++ executables: we do want them
 *
 * So for 1, everyone must include <ompi_config.h> first.  For 2, the
 * C++ bindings will never include <ompi_config.h> -- they will only
 * include <mpi.h>, which includes <ompi_config.h>, but after
 * setting OMPI_BUILDING to 0  For 3, it's the same as 1 -- just include
 * <ompi_config.h> first.
 *
 * Give code that needs to include ompi_config.h but really can't have
 * this stuff enabled (like the memory manager code) a way to turn us
 * off
 */
#if OCOMS_ENABLE_MEM_DEBUG && !defined(OCOMS_DISABLE_ENABLE_MEM_DEBUG)

/* It is safe to include ocoms/util/malloc.h here because a) it will only
   happen when we are building OMPI and therefore have a full OMPI
   source tree [including headers] available, and b) we guaranteed to
   *not* to include anything else via ocoms/util/malloc.h, so we won't
   have Cascading Includes Of Death. */
#    include "ocoms/util/malloc.h"
#    if defined(malloc)
#        undef malloc
#    endif
#    define malloc(size) ocoms_malloc((size), __FILE__, __LINE__)
#    if defined(calloc)
#        undef calloc
#    endif
#    define calloc(nmembers, size) ocoms_calloc((nmembers), (size), __FILE__, __LINE__)
#    if defined(realloc)
#        undef realloc
#    endif
#    define realloc(ptr, size) ocoms_realloc((ptr), (size), __FILE__, __LINE__)
#    if defined(free)
#        undef free
#    endif
#    define free(ptr) ocoms_free((ptr), __FILE__, __LINE__)

/*
 * If we're mem debugging, make the OCOMS_DEBUG_ZERO resolve to memset
 */
#    include <string.h>
#    define OCOMS_DEBUG_ZERO(obj) memset(&(obj), 0, sizeof(obj))
#else
#    define OCOMS_DEBUG_ZERO(obj)
#endif

/*
 * printf functions for portability (only when building Open MPI)
 */
#if !defined(HAVE_VASPRINTF) || !defined(HAVE_VSNPRINTF)
#include <stdarg.h>
#include <stdlib.h>
#endif

#if !defined(HAVE_ASPRINTF) || !defined(HAVE_SNPRINTF) || !defined(HAVE_VASPRINTF) || !defined(HAVE_VSNPRINTF)
#include "ocoms/util/printf.h"
#endif

#ifndef HAVE_ASPRINTF
# define asprintf ocoms_asprintf
#endif

#ifndef HAVE_SNPRINTF
# define snprintf ocoms_snprintf
#endif

#ifndef HAVE_VASPRINTF
# define vasprintf ocoms_vasprintf
#endif

#ifndef HAVE_VSNPRINTF
# define vsnprintf ocoms_vsnprintf
#endif

/*
 * Some platforms (Solaris) have a broken qsort implementation.  Work
 * around by using our own.
 */
#if OCOMS_HAVE_BROKEN_QSORT
#ifdef qsort
#undef qsort
#endif

#include "ocoms/util/qsort.h"
#define qsort ocoms_qsort
#endif

/*
 * On some homogenous big-iron machines (Sandia's Red Storm), there
 * are no htonl and friends.  If that's the case, provide stubs.  I
 * would hope we never find a platform that doesn't have these macros
 * and would want to talk to the outside world... On other platforms
 * we fail to detect them correctly.
 */
#if !defined(HAVE_UNIX_BYTESWAP)
static inline uint32_t htonl(uint32_t hostvar) { return hostvar; }
static inline uint32_t ntohl(uint32_t netvar) { return netvar; }
static inline uint16_t htons(uint16_t hostvar) { return hostvar; }
static inline uint16_t ntohs(uint16_t netvar) { return netvar; }
#endif

/*
 * Define __func__-preprocessor directive if the compiler does not
 * already define it.  Define it to __FILE__ so that we at least have
 * a clue where the developer is trying to indicate where the error is
 * coming from (assuming that __func__ is typically used for
 * printf-style debugging).
 */
#if defined(HAVE_DECL___FUNC__) && !HAVE_DECL___FUNC__
#define __func__ __FILE__
#endif

#define IOVBASE_TYPE  void

/**
 * If we generate our own bool type, we need a special way to cast the result
 * in such a way to keep the compilers silent.
 */
#  define OCOMS_INT_TO_BOOL(VALUE)  (bool)(VALUE)

/**
 * Top level define to check 2 things: a) if we want ipv6 support, and
 * b) the underlying system supports ipv6.  Having one #define for
 * this makes it simpler to check throughout the code base.
 */
#if OCOMS_ENABLE_IPV6 && defined(HAVE_STRUCT_SOCKADDR_IN6)
#define OCOMS_ENABLE_IPV6 1
#else
#define OCOMS_ENABLE_IPV6 0
#endif

#if !defined(HAVE_STRUCT_SOCKADDR_STORAGE) && defined(HAVE_STRUCT_SOCKADDR_IN)
#define sockaddr_storage sockaddr
#define ss_family sa_family
#endif

/* Compatibility structure so that we don't have to have as many
   #if checks in the code base */
#if !defined(HAVE_STRUCT_SOCKADDR_IN6) && defined(HAVE_STRUCT_SOCKADDR_IN)
#define sockaddr_in6 sockaddr_in
#define sin6_len sin_len
#define sin6_family sin_family
#define sin6_port sin_port
#define sin6_addr sin_addr
#endif

#if !HAVE_DECL_AF_UNSPEC
#define AF_UNSPEC 0
#endif
#if !HAVE_DECL_PF_UNSPEC
#define PF_UNSPEC 0
#endif
#if !HAVE_DECL_AF_INET6
#define AF_INET6 AF_UNSPEC
#endif
#if !HAVE_DECL_PF_INET6
#define PF_INET6 PF_UNSPEC
#endif

#if defined(__APPLE__) && defined(HAVE_INTTYPES_H)
/* Prior to Mac OS X 10.3, the length modifier "ll" wasn't
   supported, but "q" was for long long.  This isn't ANSI
   C and causes a warning when using PRI?64 macros.  We
   don't support versions prior to OS X 10.3, so we dont'
   need such backward compatibility.  Instead, redefine
   the macros to be "ll", which is ANSI C and doesn't
   cause a compiler warning. */
#include <inttypes.h>
#if defined(__PRI_64_LENGTH_MODIFIER__)
#undef __PRI_64_LENGTH_MODIFIER__
#define __PRI_64_LENGTH_MODIFIER__ "ll"
#endif
#if defined(__SCN_64_LENGTH_MODIFIER__)
#undef __SCN_64_LENGTH_MODIFIER__
#define __SCN_64_LENGTH_MODIFIER__ "ll"
#endif
#endif

#ifdef MCS_VXWORKS
/* VXWorks puts some common functions in oddly named headers.  Rather
   than update all the places the functions are used, which would be a
   maintenance disatster, just update here... */
#ifdef HAVE_IOLIB_H
/* pipe(), ioctl() */
#include <ioLib.h>
#endif
#ifdef HAVE_SOCKLIB_H
/* socket() */
#include <sockLib.h>
#endif
#ifdef HAVE_HOSTLIB_H
/* gethostname() */
#include <hostLib.h>

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif
#endif

#endif

/* If we're in C++, then just undefine restrict and then define it to
   nothing.  "restrict" is not part of the C++ language, and we don't
   have a corresponding AC_CXX_RESTRICT to figure out what the C++
   compiler supports. */
#if defined(c_plusplus) || defined(__cplusplus)
#undef restrict
#define restrict
#endif

#else

/* For a similar reason to what is listed in ocoms_config_top.h, we
   want to protect others from the autoconf/automake-generated
   PACKAGE_<foo> macros in ocoms_config.h.  We can't put these undef's
   directly in ocoms_config.h because they'll be turned into #defines'
   via autoconf.  

   So put them here in case any only else includes OMPI/ORTE/OCOMS's
   config.h files. */

#undef PACKAGE_BUGREPORT
#undef PACKAGE_NAME
#undef PACKAGE_STRING
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION
#undef PACKAGE_URL
#undef HAVE_CONFIG_H

#endif /* OMPI_BUILDING */
