﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ivs-realtime/IvsrealtimeRequest.h>
#include <aws/ivs-realtime/Ivsrealtime_EXPORTS.h>
#include <aws/ivs-realtime/model/ParticipantTokenCapability.h>

#include <utility>

namespace Aws {
namespace ivsrealtime {
namespace Model {

/**
 */
class CreateParticipantTokenRequest : public IvsrealtimeRequest {
 public:
  AWS_IVSREALTIME_API CreateParticipantTokenRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateParticipantToken"; }

  AWS_IVSREALTIME_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>ARN of the stage to which this token is scoped.</p>
   */
  inline const Aws::String& GetStageArn() const { return m_stageArn; }
  inline bool StageArnHasBeenSet() const { return m_stageArnHasBeenSet; }
  template <typename StageArnT = Aws::String>
  void SetStageArn(StageArnT&& value) {
    m_stageArnHasBeenSet = true;
    m_stageArn = std::forward<StageArnT>(value);
  }
  template <typename StageArnT = Aws::String>
  CreateParticipantTokenRequest& WithStageArn(StageArnT&& value) {
    SetStageArn(std::forward<StageArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Duration (in minutes), after which the token expires. Default: 720 (12
   * hours).</p>
   */
  inline int GetDuration() const { return m_duration; }
  inline bool DurationHasBeenSet() const { return m_durationHasBeenSet; }
  inline void SetDuration(int value) {
    m_durationHasBeenSet = true;
    m_duration = value;
  }
  inline CreateParticipantTokenRequest& WithDuration(int value) {
    SetDuration(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Name that can be specified to help identify the token. This can be any UTF-8
   * encoded text. <i>This field is exposed to all stage participants and should not
   * be used for personally identifying, confidential, or sensitive information.</i>
   * </p>
   */
  inline const Aws::String& GetUserId() const { return m_userId; }
  inline bool UserIdHasBeenSet() const { return m_userIdHasBeenSet; }
  template <typename UserIdT = Aws::String>
  void SetUserId(UserIdT&& value) {
    m_userIdHasBeenSet = true;
    m_userId = std::forward<UserIdT>(value);
  }
  template <typename UserIdT = Aws::String>
  CreateParticipantTokenRequest& WithUserId(UserIdT&& value) {
    SetUserId(std::forward<UserIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Application-provided attributes to encode into the token and attach to a
   * stage. Map keys and values can contain UTF-8 encoded text. The maximum length of
   * this field is 1 KB total. <i>This field is exposed to all stage participants and
   * should not be used for personally identifying, confidential, or sensitive
   * information.</i> </p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetAttributes() const { return m_attributes; }
  inline bool AttributesHasBeenSet() const { return m_attributesHasBeenSet; }
  template <typename AttributesT = Aws::Map<Aws::String, Aws::String>>
  void SetAttributes(AttributesT&& value) {
    m_attributesHasBeenSet = true;
    m_attributes = std::forward<AttributesT>(value);
  }
  template <typename AttributesT = Aws::Map<Aws::String, Aws::String>>
  CreateParticipantTokenRequest& WithAttributes(AttributesT&& value) {
    SetAttributes(std::forward<AttributesT>(value));
    return *this;
  }
  template <typename AttributesKeyT = Aws::String, typename AttributesValueT = Aws::String>
  CreateParticipantTokenRequest& AddAttributes(AttributesKeyT&& key, AttributesValueT&& value) {
    m_attributesHasBeenSet = true;
    m_attributes.emplace(std::forward<AttributesKeyT>(key), std::forward<AttributesValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Set of capabilities that the user is allowed to perform in the stage.
   * Default: <code>PUBLISH, SUBSCRIBE</code>.</p>
   */
  inline const Aws::Vector<ParticipantTokenCapability>& GetCapabilities() const { return m_capabilities; }
  inline bool CapabilitiesHasBeenSet() const { return m_capabilitiesHasBeenSet; }
  template <typename CapabilitiesT = Aws::Vector<ParticipantTokenCapability>>
  void SetCapabilities(CapabilitiesT&& value) {
    m_capabilitiesHasBeenSet = true;
    m_capabilities = std::forward<CapabilitiesT>(value);
  }
  template <typename CapabilitiesT = Aws::Vector<ParticipantTokenCapability>>
  CreateParticipantTokenRequest& WithCapabilities(CapabilitiesT&& value) {
    SetCapabilities(std::forward<CapabilitiesT>(value));
    return *this;
  }
  inline CreateParticipantTokenRequest& AddCapabilities(ParticipantTokenCapability value) {
    m_capabilitiesHasBeenSet = true;
    m_capabilities.push_back(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_stageArn;

  int m_duration{0};

  Aws::String m_userId;

  Aws::Map<Aws::String, Aws::String> m_attributes;

  Aws::Vector<ParticipantTokenCapability> m_capabilities;
  bool m_stageArnHasBeenSet = false;
  bool m_durationHasBeenSet = false;
  bool m_userIdHasBeenSet = false;
  bool m_attributesHasBeenSet = false;
  bool m_capabilitiesHasBeenSet = false;
};

}  // namespace Model
}  // namespace ivsrealtime
}  // namespace Aws
