﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/neptune/NeptuneRequest.h>
#include <aws/neptune/Neptune_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Neptune {
namespace Model {

/**
 */
class SwitchoverGlobalClusterRequest : public NeptuneRequest {
 public:
  AWS_NEPTUNE_API SwitchoverGlobalClusterRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "SwitchoverGlobalCluster"; }

  AWS_NEPTUNE_API Aws::String SerializePayload() const override;

 protected:
  AWS_NEPTUNE_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The identifier of the global database cluster to switch over. This parameter
   * isn't case-sensitive.</p> <p>Constraints: Must match the identifier of an
   * existing global database cluster.</p>
   */
  inline const Aws::String& GetGlobalClusterIdentifier() const { return m_globalClusterIdentifier; }
  inline bool GlobalClusterIdentifierHasBeenSet() const { return m_globalClusterIdentifierHasBeenSet; }
  template <typename GlobalClusterIdentifierT = Aws::String>
  void SetGlobalClusterIdentifier(GlobalClusterIdentifierT&& value) {
    m_globalClusterIdentifierHasBeenSet = true;
    m_globalClusterIdentifier = std::forward<GlobalClusterIdentifierT>(value);
  }
  template <typename GlobalClusterIdentifierT = Aws::String>
  SwitchoverGlobalClusterRequest& WithGlobalClusterIdentifier(GlobalClusterIdentifierT&& value) {
    SetGlobalClusterIdentifier(std::forward<GlobalClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the secondary Neptune DB cluster that you
   * want to promote to primary for the global database.</p>
   */
  inline const Aws::String& GetTargetDbClusterIdentifier() const { return m_targetDbClusterIdentifier; }
  inline bool TargetDbClusterIdentifierHasBeenSet() const { return m_targetDbClusterIdentifierHasBeenSet; }
  template <typename TargetDbClusterIdentifierT = Aws::String>
  void SetTargetDbClusterIdentifier(TargetDbClusterIdentifierT&& value) {
    m_targetDbClusterIdentifierHasBeenSet = true;
    m_targetDbClusterIdentifier = std::forward<TargetDbClusterIdentifierT>(value);
  }
  template <typename TargetDbClusterIdentifierT = Aws::String>
  SwitchoverGlobalClusterRequest& WithTargetDbClusterIdentifier(TargetDbClusterIdentifierT&& value) {
    SetTargetDbClusterIdentifier(std::forward<TargetDbClusterIdentifierT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_globalClusterIdentifier;

  Aws::String m_targetDbClusterIdentifier;
  bool m_globalClusterIdentifierHasBeenSet = false;
  bool m_targetDbClusterIdentifierHasBeenSet = false;
};

}  // namespace Model
}  // namespace Neptune
}  // namespace Aws
