﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/redshift/RedshiftRequest.h>
#include <aws/redshift/Redshift_EXPORTS.h>
#include <aws/redshift/model/ScheduledActionFilter.h>
#include <aws/redshift/model/ScheduledActionTypeValues.h>

#include <utility>

namespace Aws {
namespace Redshift {
namespace Model {

/**
 */
class DescribeScheduledActionsRequest : public RedshiftRequest {
 public:
  AWS_REDSHIFT_API DescribeScheduledActionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeScheduledActions"; }

  AWS_REDSHIFT_API Aws::String SerializePayload() const override;

 protected:
  AWS_REDSHIFT_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the scheduled action to retrieve. </p>
   */
  inline const Aws::String& GetScheduledActionName() const { return m_scheduledActionName; }
  inline bool ScheduledActionNameHasBeenSet() const { return m_scheduledActionNameHasBeenSet; }
  template <typename ScheduledActionNameT = Aws::String>
  void SetScheduledActionName(ScheduledActionNameT&& value) {
    m_scheduledActionNameHasBeenSet = true;
    m_scheduledActionName = std::forward<ScheduledActionNameT>(value);
  }
  template <typename ScheduledActionNameT = Aws::String>
  DescribeScheduledActionsRequest& WithScheduledActionName(ScheduledActionNameT&& value) {
    SetScheduledActionName(std::forward<ScheduledActionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of the scheduled actions to retrieve. </p>
   */
  inline ScheduledActionTypeValues GetTargetActionType() const { return m_targetActionType; }
  inline bool TargetActionTypeHasBeenSet() const { return m_targetActionTypeHasBeenSet; }
  inline void SetTargetActionType(ScheduledActionTypeValues value) {
    m_targetActionTypeHasBeenSet = true;
    m_targetActionType = value;
  }
  inline DescribeScheduledActionsRequest& WithTargetActionType(ScheduledActionTypeValues value) {
    SetTargetActionType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The start time in UTC of the scheduled actions to retrieve. Only active
   * scheduled actions that have invocations after this time are retrieved.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  DescribeScheduledActionsRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The end time in UTC of the scheduled action to retrieve. Only active
   * scheduled actions that have invocations before this time are retrieved.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  DescribeScheduledActionsRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If true, retrieve only active scheduled actions. If false, retrieve only
   * disabled scheduled actions. </p>
   */
  inline bool GetActive() const { return m_active; }
  inline bool ActiveHasBeenSet() const { return m_activeHasBeenSet; }
  inline void SetActive(bool value) {
    m_activeHasBeenSet = true;
    m_active = value;
  }
  inline DescribeScheduledActionsRequest& WithActive(bool value) {
    SetActive(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>List of scheduled action filters. </p>
   */
  inline const Aws::Vector<ScheduledActionFilter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<ScheduledActionFilter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<ScheduledActionFilter>>
  DescribeScheduledActionsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = ScheduledActionFilter>
  DescribeScheduledActionsRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional parameter that specifies the starting point to return a set of
   * response records. When the results of a <a>DescribeScheduledActions</a> request
   * exceed the value specified in <code>MaxRecords</code>, Amazon Web Services
   * returns a value in the <code>Marker</code> field of the response. You can
   * retrieve the next set of response records by providing the returned marker value
   * in the <code>Marker</code> parameter and retrying the request. </p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeScheduledActionsRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of response records to return in each call. If the number
   * of remaining response records exceeds the specified <code>MaxRecords</code>
   * value, a value is returned in a <code>marker</code> field of the response. You
   * can retrieve the next set of records by retrying the command with the returned
   * marker value. </p> <p>Default: <code>100</code> </p> <p>Constraints: minimum 20,
   * maximum 100.</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeScheduledActionsRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_scheduledActionName;

  ScheduledActionTypeValues m_targetActionType{ScheduledActionTypeValues::NOT_SET};

  Aws::Utils::DateTime m_startTime{};

  Aws::Utils::DateTime m_endTime{};

  bool m_active{false};

  Aws::Vector<ScheduledActionFilter> m_filters;

  Aws::String m_marker;

  int m_maxRecords{0};
  bool m_scheduledActionNameHasBeenSet = false;
  bool m_targetActionTypeHasBeenSet = false;
  bool m_startTimeHasBeenSet = false;
  bool m_endTimeHasBeenSet = false;
  bool m_activeHasBeenSet = false;
  bool m_filtersHasBeenSet = false;
  bool m_markerHasBeenSet = false;
  bool m_maxRecordsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Redshift
}  // namespace Aws
