/*
 * Copyright (C) 2023, 2024, 2025 Nicolas Dato
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#include "itc.h"
#include "utils.h"

#include <semaphore.h>
#include <stdlib.h>
#include <sys/time.h>

struct itc {
	int nslots;
	void **slots;
	int inputidx;
	int outputidx;
	sem_t emptied;
	sem_t occupied;
};

itc *itc_alloc(int nslots)
{
	itc *ctx;

	if (nslots <= 0) {
		return NULL;
	}

	ctx = calloc(1, sizeof(*ctx));
	ctx->nslots = nslots;
	ctx->slots = (void **)calloc(nslots, sizeof(*ctx->slots));
	sem_init(&ctx->emptied, 0, nslots);
	sem_init(&ctx->occupied, 0, 0);

	return ctx;
}

void itc_free(itc **ctx, itc_free_element free_element)
{
	if (ctx == NULL || *ctx == NULL) {
		return;
	}

	itc_discard_all(*ctx, free_element);

	sem_destroy(&(*ctx)->emptied);
	sem_destroy(&(*ctx)->occupied);
	free((void *)(*ctx)->slots);
	free(*ctx);
	*ctx = NULL;
}

void *itc_retrieve(itc *ctx, int timeout_ms)
{
	struct timespec ts;
	void *element;

	if (ctx == NULL) {
		return NULL;
	}

	timespec_add_ms(get_timespec(&ts), timeout_ms);

	if (sem_timedwait(&ctx->occupied, &ts)) {
		return NULL;
	}
	element = ctx->slots[ctx->outputidx];
	ctx->outputidx = (ctx->outputidx + 1) % ctx->nslots;
	sem_post(&ctx->emptied);

	return element;
}

int itc_inject(itc *ctx, int timeout_ms, void *element)
{
	struct timespec ts;

	if (ctx == NULL || element == NULL) {
		return -1;
	}

	timespec_add_ms(get_timespec(&ts), timeout_ms);

	if (sem_timedwait(&ctx->emptied, &ts)) {
		return -1;
	}
	ctx->slots[ctx->inputidx] = element;
	ctx->inputidx = (ctx->inputidx + 1) % ctx->nslots;
	sem_post(&ctx->occupied);

	return 0;
}

void itc_wait_empty(itc *ctx)
{
	int i;

	if (ctx == NULL) {
		return;
	}

	for (i = 0; i < ctx->nslots; i++) {
		sem_wait(&ctx->emptied);
	}
	for (i = 0; i < ctx->nslots; i++) {
		sem_post(&ctx->emptied);
	}
}

void itc_discard_all(itc *ctx, itc_free_element free_element)
{
	void *element;

	if (ctx == NULL) {
		return;
	}

	while ((element = itc_retrieve(ctx, 0)) != NULL) {
		if (free_element != NULL) {
			free_element(element);
		}
	}
}

int itc_get_queued(itc *ctx)
{
	int val;
	if (ctx == NULL) {
		return -1;
	}
	sem_getvalue(&ctx->occupied, &val);
	return val;
}

int itc_get_slots(itc *ctx)
{
	if (ctx == NULL) {
		return -1;
	}
	return ctx->nslots;
}

