﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

#nullable disable

using System;
using System.ComponentModel.Composition;
using System.Diagnostics.CodeAnalysis;
using System.Windows.Media;
using Microsoft.CodeAnalysis.Classification;
using Microsoft.CodeAnalysis.Host.Mef;
using Microsoft.VisualStudio.Language.StandardClassification;
using Microsoft.VisualStudio.Text.Classification;
using Microsoft.VisualStudio.Utilities;

namespace Microsoft.CodeAnalysis.Editor.Implementation.Classification;

internal sealed class ClassificationTypeFormatDefinitions
{
    #region Preprocessor Text
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.PreprocessorText)]
    [Name(ClassificationTypeNames.PreprocessorText)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class PreprocessorTextFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public PreprocessorTextFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.Preprocessor_Text;
            this.ForegroundColor = Colors.Black;
        }
    }
    #endregion
    #region Punctuation
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.Punctuation)]
    [Name(ClassificationTypeNames.Punctuation)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class PunctuationFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public PunctuationFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.Punctuation;
            this.ForegroundColor = Colors.Black;
        }
    }
    #endregion
    #region String - Verbatim

    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.VerbatimStringLiteral)]
    [Name(ClassificationTypeNames.VerbatimStringLiteral)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class StringVerbatimFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public StringVerbatimFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.String_Verbatim;
            this.ForegroundColor = Colors.Maroon;
        }
    }

    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.StringEscapeCharacter)]
    [Name(ClassificationTypeNames.StringEscapeCharacter)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class StringEscapeCharacterFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public StringEscapeCharacterFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.String_Escape_Character;
            this.ForegroundColor = Color.FromRgb(0x9e, 0x5b, 0x71);
        }
    }

    #endregion
    #region Keyword - Control
    // Keyword - Control is ordered after Keyword to ensure this more specific
    // classification will take precedence.
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.ControlKeyword)]
    [Name(ClassificationTypeNames.ControlKeyword)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class ControlKeywordFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public ControlKeywordFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.Keyword_Control;
    }
    #endregion
    #region Operator - Overloaded
    // Operator - Overloaded is ordered after Operator to ensure this more specific
    // classification will take precedence.
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.OperatorOverloaded)]
    [Name(ClassificationTypeNames.OperatorOverloaded)]
    [Order(After = PredefinedClassificationTypeNames.Operator)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class OperatorOverloadedFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public OperatorOverloadedFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.Operator_Overloaded;
    }
    #endregion

    #region Reassigned Variable
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.ReassignedVariable)]
    [Name(ClassificationTypeNames.ReassignedVariable)]
    [Order(After = Priority.High)]
    [UserVisible(false)]
    [ExcludeFromCodeCoverage]
    private sealed class ReassignedVariableFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public ReassignedVariableFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.Reassigned_variable;
            this.TextDecorations = System.Windows.TextDecorations.Underline;
        }
    }
    #endregion

    #region Obsolete Symobl
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.ObsoleteSymbol)]
    [Name(ClassificationTypeNames.ObsoleteSymbol)]
    [Order(After = Priority.High)]
    [UserVisible(false)]
    [ExcludeFromCodeCoverage]
    private sealed class ObsoleteSymbolFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public ObsoleteSymbolFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.Obsolete_symbol;
            this.TextDecorations = System.Windows.TextDecorations.Strikethrough;
        }
    }
    #endregion

    #region Symbol - Static
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.StaticSymbol)]
    [Name(ClassificationTypeNames.StaticSymbol)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class SymbolStaticFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public SymbolStaticFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.Symbol_Static;
            // The static classification is intended to be an additive classification
            // that simply changes the font's styling (bold or not). Allowing
            // customization of the foreground color would cause issues with
            // TaggedText as it is currently implemented, since any particular
            // span can only be tagged with a single TextTag.

            // By restricting to only font style, the QuickInfo and FAR render with the
            // colors the user would expect. The missing font style is not an problem
            // for these experiences because the QuickInfo already renders the static
            // modifier as part of its text and the FAR window already applies its
            // own bolding to the rendered output.
            this.BackgroundCustomizable = false;
            this.ForegroundCustomizable = false;
        }
    }
    #endregion

    // User Types - * and User Members - * are ordered after Keyword
    // so that, in the case both classifications are applied to the same
    // span, the styling for the identifier type would be chosen.

    // User Types - * and User Members - * are ordered before Symbol - Static
    // so that the font styling chosen for static symbols would override the
    // styling chosen for specific identifier types.
    #region User Types - Classes
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.ClassName)]
    [Name(ClassificationTypeNames.ClassName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [Order(Before = ClassificationTypeNames.StaticSymbol)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserTypeClassesFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserTypeClassesFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.User_Types_Classes;
            this.ForegroundColor = Color.FromRgb(0x2B, 0x91, 0xAF);
        }
    }
    #endregion
    #region User Types - Records
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.RecordClassName)]
    [Name(ClassificationTypeNames.RecordClassName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [Order(Before = ClassificationTypeNames.StaticSymbol)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserTypeRecordsFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserTypeRecordsFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.User_Types_Records;
        }
    }
    #endregion
    #region User Types - Record structs
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.RecordStructName)]
    [Name(ClassificationTypeNames.RecordStructName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [Order(Before = ClassificationTypeNames.StaticSymbol)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserTypeRecordStructsFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserTypeRecordStructsFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.User_Types_Record_Structs;
        }
    }
    #endregion
    #region User Types - Delegates
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.DelegateName)]
    [Name(ClassificationTypeNames.DelegateName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [Order(Before = ClassificationTypeNames.StaticSymbol)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserTypeDelegatesFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserTypeDelegatesFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.User_Types_Delegates;
            this.ForegroundColor = Color.FromRgb(0x2B, 0x91, 0xAF);
        }
    }
    #endregion
    #region User Types - Enums
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.EnumName)]
    [Name(ClassificationTypeNames.EnumName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [Order(Before = ClassificationTypeNames.StaticSymbol)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserTypeEnumsFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserTypeEnumsFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.User_Types_Enums;
            this.ForegroundColor = Color.FromRgb(0x2B, 0x91, 0xAF);
        }
    }
    #endregion
    #region User Types - Interfaces
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.InterfaceName)]
    [Name(ClassificationTypeNames.InterfaceName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [Order(Before = ClassificationTypeNames.StaticSymbol)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserTypeInterfacesFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserTypeInterfacesFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.User_Types_Interfaces;
            this.ForegroundColor = Color.FromRgb(0x2B, 0x91, 0xAF);
        }
    }
    #endregion
    #region User Types - Modules
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.ModuleName)]
    [Name(ClassificationTypeNames.ModuleName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [Order(Before = ClassificationTypeNames.StaticSymbol)]
    [UserVisible(true)]
    private sealed class UserTypeModulesFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserTypeModulesFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.User_Types_Modules;
            this.ForegroundColor = Color.FromRgb(0x2B, 0x91, 0xAF);
        }
    }
    #endregion
    #region User Types - Structures
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.StructName)]
    [Name(ClassificationTypeNames.StructName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [Order(Before = ClassificationTypeNames.StaticSymbol)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserTypeStructuresFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserTypeStructuresFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.User_Types_Structures;
            this.ForegroundColor = Color.FromRgb(0x2B, 0x91, 0xAF);
        }
    }
    #endregion
    #region User Types - Type Parameters
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.TypeParameterName)]
    [Name(ClassificationTypeNames.TypeParameterName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [Order(Before = ClassificationTypeNames.StaticSymbol)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserTypeTypeParametersFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserTypeTypeParametersFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.User_Types_Type_Parameters;
            this.ForegroundColor = Color.FromRgb(0x2B, 0x91, 0xAF);
        }
    }
    #endregion

    #region User Members - Fields
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.FieldName)]
    [Name(ClassificationTypeNames.FieldName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [Order(Before = ClassificationTypeNames.StaticSymbol)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserMembersFieldNameFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserMembersFieldNameFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.User_Members_Fields;
    }
    #endregion
    #region User Members - Enum Members
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.EnumMemberName)]
    [Name(ClassificationTypeNames.EnumMemberName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [Order(Before = ClassificationTypeNames.StaticSymbol)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserMembersEnumMemberNameFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserMembersEnumMemberNameFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.User_Members_Enum_Members;
    }
    #endregion
    #region User Members - Constants
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.ConstantName)]
    [Name(ClassificationTypeNames.ConstantName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [Order(Before = ClassificationTypeNames.StaticSymbol)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserMembersConstantNameFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserMembersConstantNameFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.User_Members_Constants;
    }
    #endregion
    #region User Members - Locals
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.LocalName)]
    [Name(ClassificationTypeNames.LocalName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [Order(Before = ClassificationTypeNames.StaticSymbol)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserMembersLocalNameFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserMembersLocalNameFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.User_Members_Locals;
    }
    #endregion
    #region User Members - Parameters
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.ParameterName)]
    [Name(ClassificationTypeNames.ParameterName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [Order(Before = ClassificationTypeNames.StaticSymbol)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserMembersParameterNameFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserMembersParameterNameFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.User_Members_Parameters;
    }
    #endregion
    #region User Members - Methods
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.MethodName)]
    [Name(ClassificationTypeNames.MethodName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [Order(Before = ClassificationTypeNames.StaticSymbol)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserMembersMethodNameFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserMembersMethodNameFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.User_Members_Methods;
    }
    #endregion
    #region User Members - Extension Methods
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.ExtensionMethodName)]
    [Name(ClassificationTypeNames.ExtensionMethodName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [Order(Before = ClassificationTypeNames.StaticSymbol)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserMembersExtensionMethodNameFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserMembersExtensionMethodNameFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.User_Members_Extension_Methods;
    }
    #endregion
    #region User Members - Properties
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.PropertyName)]
    [Name(ClassificationTypeNames.PropertyName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [Order(Before = ClassificationTypeNames.StaticSymbol)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserMembersPropertyNameFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserMembersPropertyNameFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.User_Members_Properties;
    }
    #endregion
    #region User Members - Events
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.EventName)]
    [Name(ClassificationTypeNames.EventName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserMembersEventNameFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserMembersEventNameFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.User_Members_Events;
    }
    #endregion
    #region User Members - Namespaces
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.NamespaceName)]
    [Name(ClassificationTypeNames.NamespaceName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserMembersNamespaceNameFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserMembersNamespaceNameFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.User_Members_Namespaces;
    }
    #endregion
    #region User Members - Labels
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.LabelName)]
    [Name(ClassificationTypeNames.LabelName)]
    [Order(After = PredefinedClassificationTypeNames.Identifier)]
    [Order(After = PredefinedClassificationTypeNames.Keyword)]
    [Order(Before = ClassificationTypeNames.StaticSymbol)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class UserMembersLabelNameFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public UserMembersLabelNameFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.User_Members_Labels;
    }
    #endregion

    #region XML Doc Comments - Attribute Name
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlDocCommentAttributeName)]
    [Name(ClassificationTypeNames.XmlDocCommentAttributeName)]
    [Order(After = Priority.Default, Before = Priority.High)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class XmlDocCommentAttributeNameFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlDocCommentAttributeNameFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.XML_Doc_Comments_Attribute_Name;
            this.ForegroundColor = Color.FromRgb(0x80, 0x80, 0x80); // CIDARKGRAY
        }
    }
    #endregion
    #region XML Doc Comments - Attribute Quotes
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlDocCommentAttributeQuotes)]
    [Name(ClassificationTypeNames.XmlDocCommentAttributeQuotes)]
    [Order(After = Priority.Default, Before = Priority.High)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class XmlDocCommentAttributeQuotesFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlDocCommentAttributeQuotesFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.XML_Doc_Comments_Attribute_Quotes;
            this.ForegroundColor = Color.FromRgb(0x80, 0x80, 0x80); // CIDARKGRAY
        }
    }
    #endregion
    #region XML Doc Comments - Attribute Value
    // definition of how format is represented in tools options.
    // also specifies the default format.
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlDocCommentAttributeValue)]
    [Name(ClassificationTypeNames.XmlDocCommentAttributeValue)]
    [Order(After = Priority.Default, Before = Priority.High)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class XmlDocCommentAttributeValueFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlDocCommentAttributeValueFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.XML_Doc_Comments_Attribute_Value;
            this.ForegroundColor = Color.FromRgb(0x80, 0x80, 0x80); // CIDARKGRAY
        }
    }
    #endregion
    #region XML Doc Comments - CData Section
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlDocCommentCDataSection)]
    [Name(ClassificationTypeNames.XmlDocCommentCDataSection)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class XmlDocCommentCDataSectionFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlDocCommentCDataSectionFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.XML_Doc_Comments_CData_Section;
            this.ForegroundColor = Color.FromRgb(0x80, 0x80, 0x80);    // CIDARKGRAY
        }
    }
    #endregion
    #region XML Doc Comments - Comment
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlDocCommentComment)]
    [Name(ClassificationTypeNames.XmlDocCommentComment)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class XmlDocCommentCommentFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlDocCommentCommentFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.XML_Doc_Comments_Comment;
            this.ForegroundColor = Color.FromRgb(0x80, 0x80, 0x80);    // CIDARKGRAY
        }
    }
    #endregion
    #region XML Doc Comments - Delimiter
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlDocCommentDelimiter)]
    [Name(ClassificationTypeNames.XmlDocCommentDelimiter)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class XmlDocCommentDelimiterFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlDocCommentDelimiterFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.XML_Doc_Comments_Delimiter;
            this.ForegroundColor = Color.FromRgb(0x80, 0x80, 0x80); // CIDARKGRAY
        }
    }
    #endregion
    #region XML Doc Comments - Entity Reference
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlDocCommentEntityReference)]
    [Name(ClassificationTypeNames.XmlDocCommentEntityReference)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class XmlDocCommentEntityReferenceFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlDocCommentEntityReferenceFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.XML_Doc_Comments_Entity_Reference;
            this.ForegroundColor = Colors.Green;
        }
    }
    #endregion
    #region XML Doc Comments - Name
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlDocCommentName)]
    [Name(ClassificationTypeNames.XmlDocCommentName)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class XmlDocCommentNameFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlDocCommentNameFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.XML_Doc_Comments_Name;
            this.ForegroundColor = Color.FromRgb(0x80, 0x80, 0x80); // CIDARKGRAY
        }
    }
    #endregion
    #region XML Doc Comments - Processing Instruction
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlDocCommentProcessingInstruction)]
    [Name(ClassificationTypeNames.XmlDocCommentProcessingInstruction)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class XmlDocCommentProcessingInstructionFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlDocCommentProcessingInstructionFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.XML_Doc_Comments_Processing_Instruction;
            this.ForegroundColor = Color.FromRgb(0x80, 0x80, 0x80); // CIDARKGRAY
        }
    }
    #endregion
    #region XML Doc Comments - Text
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlDocCommentText)]
    [Name(ClassificationTypeNames.XmlDocCommentText)]
    [Order(After = Priority.Default, Before = Priority.High)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class XmlDocCommentTextFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlDocCommentTextFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.XML_Doc_Comments_Text;
            this.ForegroundColor = Colors.Green;
        }
    }
    #endregion

    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.TestCode)]
    [Name(ClassificationTypeNames.TestCode)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class TestCodeFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public TestCodeFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.Roslyn_Test_Code;
            this.BackgroundColor = Color.FromRgb(0xe5, 0xe5, 0xe5);
        }
    }

    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.TestCodeMarkdown)]
    [Name(ClassificationTypeNames.TestCodeMarkdown)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class TestCodeMarkdownFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public TestCodeMarkdownFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.Roslyn_Test_Code_Markup;
            this.ForegroundColor = Color.FromRgb(0xff, 0x00, 0xc1);
        }
    }

    #region Regex

    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.RegexComment)]
    [Name(ClassificationTypeNames.RegexComment)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class RegexCommentFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public RegexCommentFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.Regex_Comment;
            this.ForegroundColor = Color.FromRgb(0x00, 0x80, 0x00);
        }
    }

    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.RegexCharacterClass)]
    [Name(ClassificationTypeNames.RegexCharacterClass)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class RegexCharacterClassFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public RegexCharacterClassFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.Regex_Character_Class;
            this.ForegroundColor = Color.FromRgb(0x00, 0x73, 0xff);
        }
    }

    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.RegexAnchor)]
    [Name(ClassificationTypeNames.RegexAnchor)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class RegexAnchorFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public RegexAnchorFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.Regex_Anchor;
            this.ForegroundColor = Color.FromRgb(0xff, 0x00, 0xc1);
        }
    }

    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.RegexQuantifier)]
    [Name(ClassificationTypeNames.RegexQuantifier)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class RegexQuantifierFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public RegexQuantifierFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.Regex_Quantifier;
            this.ForegroundColor = Color.FromRgb(0xff, 0x00, 0xc1);
        }
    }

    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.RegexGrouping)]
    [Name(ClassificationTypeNames.RegexGrouping)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class RegexGroupingFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public RegexGroupingFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.Regex_Grouping;
            this.ForegroundColor = Color.FromRgb(0x05, 0xc3, 0xba);
        }
    }

    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.RegexAlternation)]
    [Name(ClassificationTypeNames.RegexAlternation)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class RegexAlternationFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public RegexAlternationFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.Regex_Alternation;
            this.ForegroundColor = Color.FromRgb(0x05, 0xc3, 0xba);
        }
    }

    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.RegexText)]
    [Name(ClassificationTypeNames.RegexText)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class RegexTextFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public RegexTextFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.Regex_Text;
            this.ForegroundColor = Color.FromRgb(0x80, 0x00, 0x00);
        }
    }

    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.RegexSelfEscapedCharacter)]
    [Name(ClassificationTypeNames.RegexSelfEscapedCharacter)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class RegexSelfEscapedCharacterFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public RegexSelfEscapedCharacterFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.Regex_SelfEscapedCharacter;

            // by default, we make a self-escaped character just the bolded form of the normal
            // text color.
            this.ForegroundColor = Color.FromRgb(0x80, 0x00, 0x00);
            this.IsBold = true;
        }
    }

    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.RegexOtherEscape)]
    [Name(ClassificationTypeNames.RegexOtherEscape)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class RegexOtherEscapeFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public RegexOtherEscapeFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.Regex_OtherEscape;
            this.ForegroundColor = Color.FromRgb(0x9e, 0x5b, 0x71);
        }
    }
    #endregion

    #region JSON

    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.JsonComment)]
    [Name(ClassificationTypeNames.JsonComment)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class JsonCommentFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public JsonCommentFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.JSON_in_string_literal_Comment;
    }
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.JsonNumber)]
    [Name(ClassificationTypeNames.JsonNumber)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class JsonNumberFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public JsonNumberFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.JSON_in_string_literal_Number;
    }
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.JsonString)]
    [Name(ClassificationTypeNames.JsonString)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class JsonStringFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public JsonStringFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.JSON_in_string_literal_String;
    }
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.JsonKeyword)]
    [Name(ClassificationTypeNames.JsonKeyword)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class JsonKeywordFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public JsonKeywordFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.JSON_in_string_literal_Keyword;
    }
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.JsonText)]
    [Name(ClassificationTypeNames.JsonText)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class JsonTextFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public JsonTextFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.JSON_in_string_literal_Text;
    }
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.JsonOperator)]
    [Name(ClassificationTypeNames.JsonOperator)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class JsonOperatorFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public JsonOperatorFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.JSON_in_string_literal_Operator;
    }
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.JsonPunctuation)]
    [Name(ClassificationTypeNames.JsonPunctuation)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class JsonPunctuationFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public JsonPunctuationFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.JSON_in_string_literal_Punctuation;
    }
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.JsonObject)]
    [Name(ClassificationTypeNames.JsonObject)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class JsonObjectFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public JsonObjectFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.JSON_in_string_literal_Object;
    }
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.JsonArray)]
    [Name(ClassificationTypeNames.JsonArray)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class JsonArrayFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public JsonArrayFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.JSON_in_string_literal_Array;
    }
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.JsonPropertyName)]
    [Name(ClassificationTypeNames.JsonPropertyName)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class JsonPropertyNameFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public JsonPropertyNameFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.JSON_in_string_literal_Property_Name;
    }
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.JsonConstructorName)]
    [Name(ClassificationTypeNames.JsonConstructorName)]
    [Order(After = ClassificationTypeNames.StringLiteral)]
    [Order(After = ClassificationTypeNames.VerbatimStringLiteral)]
    [UserVisible(true)]
    [ExcludeFromCodeCoverage]
    private sealed class JsonConstructorNameFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public JsonConstructorNameFormatDefinition()
            => this.DisplayName = EditorFeaturesResources.JSON_in_string_literal_Constructor_Name;
    }
    #endregion

    #region VB XML Literals - Attribute Name
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlLiteralAttributeName)]
    [Name(ClassificationTypeNames.XmlLiteralAttributeName)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    private sealed class XmlLiteralAttributeNameFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlLiteralAttributeNameFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.VB_XML_Literals_Attribute_Name;
            this.ForegroundColor = Color.FromRgb(0xB9, 0x64, 0x64); // HC_LIGHTRED
        }
    }
    #endregion
    #region VB XML Literals - Attribute Quotes
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlLiteralAttributeQuotes)]
    [Name(ClassificationTypeNames.XmlLiteralAttributeQuotes)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    private sealed class XmlLiteralAttributeQuotesFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlLiteralAttributeQuotesFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.VB_XML_Literals_Attribute_Quotes;
            this.ForegroundColor = Color.FromRgb(0x55, 0x55, 0x55); // HC_LIGHTBLACK
        }
    }
    #endregion
    #region VB XML Literals - Attribute Value
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlLiteralAttributeValue)]
    [Name(ClassificationTypeNames.XmlLiteralAttributeValue)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    private sealed class XmlLiteralAttributeValueFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlLiteralAttributeValueFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.VB_XML_Literals_Attribute_Value;
            this.ForegroundColor = Color.FromRgb(0x64, 0x64, 0xB9); // HC_LIGHTBLUE
        }
    }
    #endregion
    #region VB XML Literals - CData Section
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlLiteralCDataSection)]
    [Name(ClassificationTypeNames.XmlLiteralCDataSection)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    private sealed class XmlLiteralCDataSectionFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlLiteralCDataSectionFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.VB_XML_Literals_CData_Section;
            this.ForegroundColor = Color.FromRgb(0x80, 0x80, 0x80);    // CIDARKGRAY
        }
    }
    #endregion
    #region VB XML Literals - Comment
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlLiteralComment)]
    [Name(ClassificationTypeNames.XmlLiteralComment)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    private sealed class XmlLiteralCommentFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlLiteralCommentFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.VB_XML_Literals_Comment;
            this.ForegroundColor = Color.FromRgb(0x62, 0x97, 0x55); // HC_LIGHTGREEN
        }
    }
    #endregion
    #region VB XML Literals - Delimiter
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlLiteralDelimiter)]
    [Name(ClassificationTypeNames.XmlLiteralDelimiter)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    private sealed class XmlLiteralDelimiterFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlLiteralDelimiterFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.VB_XML_Literals_Delimiter;
            this.ForegroundColor = Color.FromRgb(0x64, 0x64, 0xB9); // HC_LIGHTBLUE
        }
    }
    #endregion
    #region VB XML Literals - Embedded Expression
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlLiteralEmbeddedExpression)]
    [Name(ClassificationTypeNames.XmlLiteralEmbeddedExpression)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    private sealed class XmlLiteralEmbeddedExpressionFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlLiteralEmbeddedExpressionFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.VB_XML_Literals_Embedded_Expression;
            this.ForegroundColor = Color.FromRgb(0x55, 0x55, 0x55); // HC_LIGHTBLACK
            this.BackgroundColor = Color.FromRgb(0xFF, 0xFE, 0xBF); // HC_LIGHTYELLOW
        }
    }
    #endregion
    #region VB XML Literals - Entity Reference
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlLiteralEntityReference)]
    [Name(ClassificationTypeNames.XmlLiteralEntityReference)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    private sealed class XmlLiteralEntityReferenceFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlLiteralEntityReferenceFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.VB_XML_Literals_Entity_Reference;
            this.ForegroundColor = Color.FromRgb(0xB9, 0x64, 0x64); // HC_LIGHTRED
        }
    }
    #endregion
    #region VB XML Literals - Name
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlLiteralName)]
    [Name(ClassificationTypeNames.XmlLiteralName)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    private sealed class XmlLiteralNameFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlLiteralNameFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.VB_XML_Literals_Name;
            this.ForegroundColor = Color.FromRgb(0x84, 0x46, 0x46); // HC_LIGHTMAROON
        }
    }
    #endregion
    #region VB XML Literals - Processing Instruction
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlLiteralProcessingInstruction)]
    [Name(ClassificationTypeNames.XmlLiteralProcessingInstruction)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    private sealed class XmlLiteralProcessingInstructionFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlLiteralProcessingInstructionFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.VB_XML_Literals_Processing_Instruction;
            this.ForegroundColor = Color.FromRgb(0xC0, 0xC0, 0xC0); // HC_LIGHTGRAY
        }
    }
    #endregion
    #region VB XML Literals - Text
    [Export(typeof(EditorFormatDefinition))]
    [ClassificationType(ClassificationTypeNames = ClassificationTypeNames.XmlLiteralText)]
    [Name(ClassificationTypeNames.XmlLiteralText)]
    [Order(After = LanguagePriority.NaturalLanguage, Before = LanguagePriority.FormalLanguage)]
    [UserVisible(true)]
    private sealed class XmlLiteralTextFormatDefinition : ClassificationFormatDefinition
    {
        [ImportingConstructor]
        [Obsolete(MefConstruction.ImportingConstructorMessage, error: true)]
        public XmlLiteralTextFormatDefinition()
        {
            this.DisplayName = EditorFeaturesResources.VB_XML_Literals_Text;
            this.ForegroundColor = Color.FromRgb(0x55, 0x55, 0x55); // HC_LIGHTBLACK
        }
    }
    #endregion
}
