// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import 'chrome://resources/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/cr_elements/cr_icon_button/cr_icon_button.js';
import 'chrome://resources/cr_elements/cr_icon/cr_icon.js';
import 'chrome://resources/cr_elements/icons_lit.html.js';

import type {CrButtonElement} from 'chrome://resources/cr_elements/cr_button/cr_button.js';
import {I18nMixinLit} from 'chrome://resources/cr_elements/i18n_mixin_lit.js';
import {WebUiListenerMixinLit} from 'chrome://resources/cr_elements/web_ui_listener_mixin_lit.js';
import {assert} from 'chrome://resources/js/assert.js';
import {focusWithoutInk} from 'chrome://resources/js/focus_without_ink.js';
import {loadTimeData} from 'chrome://resources/js/load_time_data.js';
import {CrLitElement} from 'chrome://resources/lit/v3_0/lit.rollup.js';

import type {AutogeneratedThemeColorInfo,
             // <if expr="chromeos_lacros">
             AvailableAccount,
             // </if>
             ManageProfilesBrowserProxy} from '../manage_profiles_browser_proxy.js';
import {ManageProfilesBrowserProxyImpl} from '../manage_profiles_browser_proxy.js';
import {
  // <if expr="chromeos_lacros">
  navigateTo,
  // </if>
  navigateToPreviousRoute, navigateToStep, ProfileCreationSteps, recordPageVisited, Routes} from '../navigation_mixin.js';
import {createDummyAutogeneratedThemeColorInfo} from '../profile_picker_util.js';

import {getCss} from './profile_type_choice.css.js';
import {getHtml} from './profile_type_choice.html.js';

export interface ProfileTypeChoiceElement {
  $: {
    backButton: CrButtonElement,
    notNowButton: CrButtonElement,
    signInButton: CrButtonElement,
  };
}

const ProfileTypeChoiceElementBase =
    WebUiListenerMixinLit(I18nMixinLit(CrLitElement));

export class ProfileTypeChoiceElement extends ProfileTypeChoiceElementBase {
  static get is() {
    return 'profile-type-choice';
  }

  static override get styles() {
    return getCss();
  }

  override render() {
    return getHtml.bind(this)();
  }

  static override get properties() {
    return {
      profileThemeInfo: {type: Object},

      /**
       * True if a new profile (local or signed-in) is being created, all
       * buttons that create a new profile are then disabled (to avoid creating
       * two profiles).
       */
      profileCreationInProgress: {type: Boolean},

      /**
       * The disclaimer for managed devices.
       */
      managedDeviceDisclaimer_: {type: Boolean},

      // <if expr="chromeos_lacros">
      /**
       * If there are no available accounts, the account selection screen is
       * skipped.
       */
      hasAvailableAccounts_: {type: Boolean},
      // </if>
    };
  }

  profileThemeInfo: AutogeneratedThemeColorInfo =
      createDummyAutogeneratedThemeColorInfo();
  profileCreationInProgress: boolean = false;
  protected managedDeviceDisclaimer_: boolean =
      loadTimeData.getString('managedDeviceDisclaimer').length > 0;
  private manageProfilesBrowserProxy_: ManageProfilesBrowserProxy =
      ManageProfilesBrowserProxyImpl.getInstance();

  // <if expr="chromeos_lacros">
  private hasAvailableAccounts_: boolean = false;

  override connectedCallback() {
    super.connectedCallback();
    this.addWebUiListener(
        'available-accounts-changed',
        (accounts: AvailableAccount[]) =>
            this.handleAvailableAccountsChanged_(accounts));
    this.manageProfilesBrowserProxy_.getAvailableAccounts();
  }
  // </if>

  override firstUpdated() {
    this.addWebUiListener(
        'load-signin-finished',
        (success: boolean) => this.handleLoadSigninFinished_(success));
    this.manageProfilesBrowserProxy_.recordSignInPromoImpression();

    this.addEventListener('view-enter-start', this.onViewEnterStart_);
  }

  private onViewEnterStart_() {
    focusWithoutInk(this.$.backButton);
  }

  protected onNotNowClick_() {
    navigateToStep(
        Routes.NEW_PROFILE, ProfileCreationSteps.LOCAL_PROFILE_CUSTOMIZATION);
  }

  protected onSignInClick_() {
    // <if expr="chromeos_lacros">
    if (this.hasAvailableAccounts_) {
      navigateTo(Routes.ACCOUNT_SELECTION_LACROS);
      return;
    }
    // </if>

    assert(!this.profileCreationInProgress);
    this.profileCreationInProgress = true;

    // Explicitly record the page visit as this step is not pushed to the
    // history stack.
    recordPageVisited(ProfileCreationSteps.LOAD_SIGNIN);
    this.manageProfilesBrowserProxy_.selectNewAccount(
        this.profileThemeInfo.color);
  }

  protected onBackClick_() {
    navigateToPreviousRoute();
  }

  private handleLoadSigninFinished_(_success: boolean) {
    // TODO(crbug.com/40911651): If failed, show some error message to inform the
    // user.
    this.profileCreationInProgress = false;
  }

  protected getBackButtonAriaLabel_(): string {
    return this.i18n(
        'backButtonAriaLabel', this.i18n('profileTypeChoiceTitle'));
  }

  // <if expr="chromeos_lacros">
  private handleAvailableAccountsChanged_(availableAccounts:
                                              AvailableAccount[]) {
    this.hasAvailableAccounts_ = availableAccounts.length > 0;
  }
  // </if>
}

declare global {
  interface HTMLElementTagNameMap {
    'profile-type-choice': ProfileTypeChoiceElement;
  }
}

customElements.define(ProfileTypeChoiceElement.is, ProfileTypeChoiceElement);
