// @HEADER
// *****************************************************************************
//        MueLu: A package for multigrid based preconditioning
//
// Copyright 2012 NTESS and the MueLu contributors.
// SPDX-License-Identifier: BSD-3-Clause
// *****************************************************************************
// @HEADER

#ifndef MUELU_GENERICRFACTORY_DEF_HPP
#define MUELU_GENERICRFACTORY_DEF_HPP

#include <Xpetra_Matrix.hpp>

#include "MueLu_GenericRFactory_decl.hpp"

#include "MueLu_FactoryBase.hpp"
#include "MueLu_PFactory.hpp"
#include "MueLu_FactoryManagerBase.hpp"
#include "MueLu_DisableMultipleCallCheck.hpp"
#include "MueLu_Monitor.hpp"

namespace MueLu {

template <class Scalar, class LocalOrdinal, class GlobalOrdinal, class Node>
RCP<const ParameterList> GenericRFactory<Scalar, LocalOrdinal, GlobalOrdinal, Node>::GetValidParameterList() const {
  RCP<ParameterList> validParamList = rcp(new ParameterList());
  validParamList->set<RCP<const FactoryBase> >("P", Teuchos::null, "Generating factory of the matrix P");
  return validParamList;
}

template <class Scalar, class LocalOrdinal, class GlobalOrdinal, class Node>
void GenericRFactory<Scalar, LocalOrdinal, GlobalOrdinal, Node>::DeclareInput(Level & /* fineLevel */, Level &coarseLevel) const {
  RCP<const FactoryBase> PFact1 = GetFactory("P");
  if (PFact1 == Teuchos::null) {
    PFact1 = coarseLevel.GetFactoryManager()->GetFactory("P");
  }
  RCP<PFactory> PFact = Teuchos::rcp_const_cast<PFactory>(rcp_dynamic_cast<const PFactory>(PFact1));
  ;

  bool rmode = PFact->isRestrictionModeSet();
  PFact->setRestrictionMode(true);  // set restriction mode

  // Force request call for PFact
  // In general, Request is only called once for each factory, since we
  // can reuse data generated by the factory. However, here we have to
  // run the code in PFact.Build again, so we have to request the
  // dependencies of PFact first!  Thedependencies are (automatically)
  // cleaned up after the second run of PFact.Build in
  // coarseLevel.Get<RCP<Matrix> >("R",PFact.get())!
  coarseLevel.DeclareDependencies(PFact.get());

  coarseLevel.DeclareInput("R", PFact.get(), this);  // we expect the prolongation operator factory to produce "R" as output
  // call declareInput is called within DeclareInput call
  PFact->setRestrictionMode(rmode);  // reset restriciton mode flag
}

template <class Scalar, class LocalOrdinal, class GlobalOrdinal, class Node>
void GenericRFactory<Scalar, LocalOrdinal, GlobalOrdinal, Node>::Build(Level & /* fineLevel */, Level &coarseLevel) const {
  FactoryMonitor m(*this, "Call prolongator factory for calculating restrictor", coarseLevel);

  RCP<const FactoryBase> PFact1 = GetFactory("P");
  if (PFact1 == Teuchos::null) {
    PFact1 = coarseLevel.GetFactoryManager()->GetFactory("P");
  }
  RCP<PFactory> PFact = Teuchos::rcp_const_cast<PFactory>(rcp_dynamic_cast<const PFactory>(PFact1));
  ;
  MueLu::DisableMultipleCallCheck check(PFact);

  // BuildR
  bool rmode = PFact->isRestrictionModeSet();
  PFact->setRestrictionMode(true);  // switch prolongator factory to restriction mode

  // PFact->Build(fineLevel, coarseLevel);  // call PFactory::Build explicitely
  RCP<Matrix> R = coarseLevel.Get<RCP<Matrix> >("R", PFact.get());

  PFact->setRestrictionMode(rmode);  // reset restriction mode flag

  Set(coarseLevel, "R", R);

}  // BuildR

}  // namespace MueLu

#define MUELU_GENERICRFACTORY_SHORT
#endif  // MUELU_GENERICRFACTORY_DEF_HPP
