#
# Copyright 2016 Ettus Research
#

# ---------------------------------------
# Gather all external parameters
# ---------------------------------------
set bd_file         $::env(BD_FILE)               ;# Absolute path to BD/Tcl file from src dir
set src_ext [file extension $bd_file]             ;# BD or Tcl file?
set part_name       $::env(PART_NAME)              ;# Full Xilinx part name
set bd_name [file rootname [file tail $bd_file]]   ;# Extract IP name
if {[info exists env(BD_IP_REPOS)]} {
    set ip_repos $::env(BD_IP_REPOS);# Any supporting IP repos
} else {
    set ip_repos {}
}
if {[info exists env(BD_HDL_SRCS)]} {
    set hdl_sources $::env(BD_HDL_SRCS);# Any supporting HDL files
} else {
    set hdl_sources {}
}

# Delete any previous output cookie file
file delete -force "$bd_file.out"
# ---------------------------------------
# Vivado Commands
# ---------------------------------------
create_project -part $part_name -in_memory
# In non-project mode, the hierarchy must be updated for the HDL source files to be
# correctly applied to the BD. See AR:
# https://www.xilinx.com/support/answers/63488.html
set_property source_mgmt_mode All [current_project]
set_property ip_repo_paths "{$ip_repos}" [current_project]
update_ip_catalog
# Add supplementary HDL sources, if they exist.
foreach src_file $hdl_sources {
    set hdl_ext [file extension $src_file ]
    if [expr [lsearch {.vhd .vhdl} $hdl_ext] >= 0] {
        puts "BUILDER: Adding VHDL    : $src_file"
        read_vhdl -library work $src_file
    } elseif [expr [lsearch {.v .sv .vh .svh} $hdl_ext] >= 0] {
        puts "BUILDER: Adding Verilog : $src_file"
        read_verilog $src_file
    } else {
        puts "BUILDER: \[WARNING\] File ignored!!!: $src_file"
    }
}
# Open .tcl or .bd design directly.
if [expr [lsearch {.tcl} $src_ext] >= 0] {
    puts "BUILDER: Generating Block Diagram from script: $bd_file"
    create_bd_design -dir . $bd_name
    source $bd_file
    report_ip_status
    puts "BUILDER: Report_ip_status done"
    set bd_file $bd_name.bd
} else {
    puts "BUILDER: Adding Block Diagram: $bd_file"
    add_files -norecurse $bd_file
    puts "BUILDER: Generating BD Target first pass..."
    generate_target all [get_files $bd_file] -force
    report_ip_status
    puts "BUILDER: Report_ip_status done"
    open_bd_design $bd_file
}
# Generate outputs.
puts "BUILDER: Generating BD Target..."
generate_target all [get_files $bd_file]
puts "BUILDER: Generate all done"

if { [get_msg_config -count -severity ERROR] == 0 } {
    # Write output cookie file
    set outfile [open "$bd_file.out" w]
    puts $outfile "This file was auto-generated by viv_generate_bd.tcl and signifies that BD generation is done."
    close $outfile
} else {
    exit 1
}

