// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

// This file is generated by a tool. See: src/Http/Http.Results/tools/ResultsOfTGenerator
using System.Reflection;
using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Http.HttpResults;
using Microsoft.AspNetCore.Http.Metadata;
using Microsoft.AspNetCore.Routing;
using Microsoft.AspNetCore.Routing.Patterns;

namespace Microsoft.AspNetCore.Http.Result;

public partial class ResultsOfTTests
{
    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    public void ResultsOfTResult1TResult2_Result_IsAssignedResult(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(),
                _ => new ChecksumResult2()
            };
        }

        // Act
        var result = MyApi(input);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
    }

    [Theory]
    [InlineData(1)]
    [InlineData(2)]
    public async Task ResultsOfTResult1TResult2_ExecuteResult_ExecutesAssignedResult(int input)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2> MyApi(int checksum)
        {
            return checksum switch
            {
                1 => new ChecksumResult1(checksum),
                _ => new ChecksumResult2(checksum)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Fact]
    public void ResultsOfTResult1TResult2_Throws_ArgumentNullException_WhenHttpContextIsNull()
    {
        // Arrange
        Results<ChecksumResult1, NoContent> MyApi()
        {
            return new ChecksumResult1(1);
        }
        HttpContext httpContext = null;

        // Act & Assert
        var result = MyApi();

        Assert.ThrowsAsync<ArgumentNullException>(async () =>
        {
            await result.ExecuteAsync(httpContext);
        });
    }

    [Fact]
    public void ResultsOfTResult1TResult2_Throws_InvalidOperationException_WhenResultIsNull()
    {
        // Arrange
        Results<ChecksumResult1, NoContent> MyApi()
        {
            return new ChecksumResult1(1);
        }
        var httpContext = GetHttpContext();

        // Act & Assert
        var result = MyApi();

        Assert.ThrowsAsync<InvalidOperationException>(async () =>
        {
            await result.ExecuteAsync(httpContext);
        });
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    public async Task ResultsOfTResult1TResult2_AcceptsIResult_AsFirstTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<IResult, ChecksumResult2> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                _ => new ChecksumResult2(2)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    public async Task ResultsOfTResult1TResult2_AcceptsIResult_AsSecondTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, IResult> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                _ => new ChecksumResult2(2)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2>))]
    [InlineData(3, typeof(ChecksumResult3))]
    public async Task ResultsOfTResult1TResult2_AcceptsNestedResultsOfT_AsFirstTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2>, ChecksumResult3> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2>)new ChecksumResult2(2),
                _ => new ChecksumResult3(3)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2>))]
    [InlineData(3, typeof(ChecksumResult3))]
    public async Task ResultsOfTResult1TResult2_AcceptsNestedResultsOfT_AsSecondTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2>, ChecksumResult3> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2>)new ChecksumResult2(2),
                _ => new ChecksumResult3(3)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Fact]
    public void ResultsOfTResult1TResult2_PopulateMetadata_PopulatesMetadataFromTypeArgsThatImplementIEndpointMetadataProvider()
    {
        // Arrange
        Results<ProvidesMetadataResult1, ProvidesMetadataResult2> MyApi() { throw new NotImplementedException(); }
        var builder = new RouteEndpointBuilder(requestDelegate: null, RoutePatternFactory.Parse("/"), order: 0);

        // Act
        PopulateMetadata<Results<ProvidesMetadataResult1, ProvidesMetadataResult2>>(((Delegate)MyApi).GetMethodInfo(), builder);

        // Assert
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult1) });
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult2) });
    }

    [Fact]
    public void ResultsOfTResult1TResult2_PopulateMetadata_Throws_ArgumentNullException_WhenMethodOrBuilderAreNull()
    {
        // Act & Assert
        Assert.Throws<ArgumentNullException>("method", () => PopulateMetadata<Results<ProvidesMetadataResult1, ProvidesMetadataResult2>>(null, new RouteEndpointBuilder(requestDelegate: null, RoutePatternFactory.Parse("/"), order: 0)));
        Assert.Throws<ArgumentNullException>("builder", () => PopulateMetadata<Results<ProvidesMetadataResult1, ProvidesMetadataResult2>>(((Delegate)GeneratedCodeIsUpToDate).GetMethodInfo(), null));
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    public void ResultsOfTResult1TResult2TResult3_Result_IsAssignedResult(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2, ChecksumResult3> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(),
                2 => new ChecksumResult2(),
                _ => new ChecksumResult3()
            };
        }

        // Act
        var result = MyApi(input);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
    }

    [Theory]
    [InlineData(1)]
    [InlineData(2)]
    [InlineData(3)]
    public async Task ResultsOfTResult1TResult2TResult3_ExecuteResult_ExecutesAssignedResult(int input)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2, ChecksumResult3> MyApi(int checksum)
        {
            return checksum switch
            {
                1 => new ChecksumResult1(checksum),
                2 => new ChecksumResult2(checksum),
                _ => new ChecksumResult3(checksum)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Fact]
    public void ResultsOfTResult1TResult2TResult3_Throws_ArgumentNullException_WhenHttpContextIsNull()
    {
        // Arrange
        Results<ChecksumResult1, NoContent> MyApi()
        {
            return new ChecksumResult1(1);
        }
        HttpContext httpContext = null;

        // Act & Assert
        var result = MyApi();

        Assert.ThrowsAsync<ArgumentNullException>(async () =>
        {
            await result.ExecuteAsync(httpContext);
        });
    }

    [Fact]
    public void ResultsOfTResult1TResult2TResult3_Throws_InvalidOperationException_WhenResultIsNull()
    {
        // Arrange
        Results<ChecksumResult1, NoContent> MyApi()
        {
            return new ChecksumResult1(1);
        }
        var httpContext = GetHttpContext();

        // Act & Assert
        var result = MyApi();

        Assert.ThrowsAsync<InvalidOperationException>(async () =>
        {
            await result.ExecuteAsync(httpContext);
        });
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    public async Task ResultsOfTResult1TResult2TResult3_AcceptsIResult_AsFirstTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<IResult, ChecksumResult2, ChecksumResult3> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                2 => new ChecksumResult2(2),
                _ => new ChecksumResult3(3)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    public async Task ResultsOfTResult1TResult2TResult3_AcceptsIResult_AsSecondTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, IResult, ChecksumResult3> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                2 => new ChecksumResult2(2),
                _ => new ChecksumResult3(3)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    public async Task ResultsOfTResult1TResult2TResult3_AcceptsIResult_AsThirdTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2, IResult> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                2 => new ChecksumResult2(2),
                _ => new ChecksumResult3(3)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>))]
    [InlineData(3, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>))]
    [InlineData(4, typeof(ChecksumResult4))]
    public async Task ResultsOfTResult1TResult2TResult3_AcceptsNestedResultsOfT_AsFirstTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>, ChecksumResult4> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>)new ChecksumResult2(2),
                3 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>)new ChecksumResult3(3),
                _ => new ChecksumResult4(4)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>))]
    [InlineData(3, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>))]
    [InlineData(4, typeof(ChecksumResult4))]
    public async Task ResultsOfTResult1TResult2TResult3_AcceptsNestedResultsOfT_AsSecondTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>, ChecksumResult4> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>)new ChecksumResult2(2),
                3 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>)new ChecksumResult3(3),
                _ => new ChecksumResult4(4)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>))]
    [InlineData(3, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>))]
    [InlineData(4, typeof(ChecksumResult4))]
    public async Task ResultsOfTResult1TResult2TResult3_AcceptsNestedResultsOfT_AsThirdTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>, ChecksumResult4> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>)new ChecksumResult2(2),
                3 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3>)new ChecksumResult3(3),
                _ => new ChecksumResult4(4)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Fact]
    public void ResultsOfTResult1TResult2TResult3_PopulateMetadata_PopulatesMetadataFromTypeArgsThatImplementIEndpointMetadataProvider()
    {
        // Arrange
        Results<ProvidesMetadataResult1, ProvidesMetadataResult2, ProvidesMetadataResult3> MyApi() { throw new NotImplementedException(); }
        var builder = new RouteEndpointBuilder(requestDelegate: null, RoutePatternFactory.Parse("/"), order: 0);

        // Act
        PopulateMetadata<Results<ProvidesMetadataResult1, ProvidesMetadataResult2, ProvidesMetadataResult3>>(((Delegate)MyApi).GetMethodInfo(), builder);

        // Assert
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult1) });
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult2) });
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult3) });
    }

    [Fact]
    public void ResultsOfTResult1TResult2TResult3_PopulateMetadata_Throws_ArgumentNullException_WhenMethodOrBuilderAreNull()
    {
        // Act & Assert
        Assert.Throws<ArgumentNullException>("method", () => PopulateMetadata<Results<ProvidesMetadataResult1, ProvidesMetadataResult2, ProvidesMetadataResult3>>(null, new RouteEndpointBuilder(requestDelegate: null, RoutePatternFactory.Parse("/"), order: 0)));
        Assert.Throws<ArgumentNullException>("builder", () => PopulateMetadata<Results<ProvidesMetadataResult1, ProvidesMetadataResult2, ProvidesMetadataResult3>>(((Delegate)GeneratedCodeIsUpToDate).GetMethodInfo(), null));
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    [InlineData(4, typeof(ChecksumResult4))]
    public void ResultsOfTResult1TResult2TResult3TResult4_Result_IsAssignedResult(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(),
                2 => new ChecksumResult2(),
                3 => new ChecksumResult3(),
                _ => new ChecksumResult4()
            };
        }

        // Act
        var result = MyApi(input);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
    }

    [Theory]
    [InlineData(1)]
    [InlineData(2)]
    [InlineData(3)]
    [InlineData(4)]
    public async Task ResultsOfTResult1TResult2TResult3TResult4_ExecuteResult_ExecutesAssignedResult(int input)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4> MyApi(int checksum)
        {
            return checksum switch
            {
                1 => new ChecksumResult1(checksum),
                2 => new ChecksumResult2(checksum),
                3 => new ChecksumResult3(checksum),
                _ => new ChecksumResult4(checksum)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Fact]
    public void ResultsOfTResult1TResult2TResult3TResult4_Throws_ArgumentNullException_WhenHttpContextIsNull()
    {
        // Arrange
        Results<ChecksumResult1, NoContent> MyApi()
        {
            return new ChecksumResult1(1);
        }
        HttpContext httpContext = null;

        // Act & Assert
        var result = MyApi();

        Assert.ThrowsAsync<ArgumentNullException>(async () =>
        {
            await result.ExecuteAsync(httpContext);
        });
    }

    [Fact]
    public void ResultsOfTResult1TResult2TResult3TResult4_Throws_InvalidOperationException_WhenResultIsNull()
    {
        // Arrange
        Results<ChecksumResult1, NoContent> MyApi()
        {
            return new ChecksumResult1(1);
        }
        var httpContext = GetHttpContext();

        // Act & Assert
        var result = MyApi();

        Assert.ThrowsAsync<InvalidOperationException>(async () =>
        {
            await result.ExecuteAsync(httpContext);
        });
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    [InlineData(4, typeof(ChecksumResult4))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4_AcceptsIResult_AsFirstTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<IResult, ChecksumResult2, ChecksumResult3, ChecksumResult4> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                2 => new ChecksumResult2(2),
                3 => new ChecksumResult3(3),
                _ => new ChecksumResult4(4)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    [InlineData(4, typeof(ChecksumResult4))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4_AcceptsIResult_AsSecondTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, IResult, ChecksumResult3, ChecksumResult4> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                2 => new ChecksumResult2(2),
                3 => new ChecksumResult3(3),
                _ => new ChecksumResult4(4)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    [InlineData(4, typeof(ChecksumResult4))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4_AcceptsIResult_AsThirdTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2, IResult, ChecksumResult4> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                2 => new ChecksumResult2(2),
                3 => new ChecksumResult3(3),
                _ => new ChecksumResult4(4)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    [InlineData(4, typeof(ChecksumResult4))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4_AcceptsIResult_AsFourthTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, IResult> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                2 => new ChecksumResult2(2),
                3 => new ChecksumResult3(3),
                _ => new ChecksumResult4(4)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>))]
    [InlineData(3, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>))]
    [InlineData(4, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>))]
    [InlineData(5, typeof(ChecksumResult5))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4_AcceptsNestedResultsOfT_AsFirstTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>, ChecksumResult5> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>)new ChecksumResult2(2),
                3 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>)new ChecksumResult3(3),
                4 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>)new ChecksumResult4(4),
                _ => new ChecksumResult5(5)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>))]
    [InlineData(3, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>))]
    [InlineData(4, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>))]
    [InlineData(5, typeof(ChecksumResult5))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4_AcceptsNestedResultsOfT_AsSecondTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>, ChecksumResult5> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>)new ChecksumResult2(2),
                3 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>)new ChecksumResult3(3),
                4 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>)new ChecksumResult4(4),
                _ => new ChecksumResult5(5)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>))]
    [InlineData(3, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>))]
    [InlineData(4, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>))]
    [InlineData(5, typeof(ChecksumResult5))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4_AcceptsNestedResultsOfT_AsThirdTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>, ChecksumResult5> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>)new ChecksumResult2(2),
                3 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>)new ChecksumResult3(3),
                4 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>)new ChecksumResult4(4),
                _ => new ChecksumResult5(5)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>))]
    [InlineData(3, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>))]
    [InlineData(4, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>))]
    [InlineData(5, typeof(ChecksumResult5))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4_AcceptsNestedResultsOfT_AsFourthTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>, ChecksumResult5> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>)new ChecksumResult2(2),
                3 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>)new ChecksumResult3(3),
                4 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4>)new ChecksumResult4(4),
                _ => new ChecksumResult5(5)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Fact]
    public void ResultsOfTResult1TResult2TResult3TResult4_PopulateMetadata_PopulatesMetadataFromTypeArgsThatImplementIEndpointMetadataProvider()
    {
        // Arrange
        Results<ProvidesMetadataResult1, ProvidesMetadataResult2, ProvidesMetadataResult3, ProvidesMetadataResult4> MyApi() { throw new NotImplementedException(); }
        var builder = new RouteEndpointBuilder(requestDelegate: null, RoutePatternFactory.Parse("/"), order: 0);

        // Act
        PopulateMetadata<Results<ProvidesMetadataResult1, ProvidesMetadataResult2, ProvidesMetadataResult3, ProvidesMetadataResult4>>(((Delegate)MyApi).GetMethodInfo(), builder);

        // Assert
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult1) });
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult2) });
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult3) });
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult4) });
    }

    [Fact]
    public void ResultsOfTResult1TResult2TResult3TResult4_PopulateMetadata_Throws_ArgumentNullException_WhenMethodOrBuilderAreNull()
    {
        // Act & Assert
        Assert.Throws<ArgumentNullException>("method", () => PopulateMetadata<Results<ProvidesMetadataResult1, ProvidesMetadataResult2, ProvidesMetadataResult3, ProvidesMetadataResult4>>(null, new RouteEndpointBuilder(requestDelegate: null, RoutePatternFactory.Parse("/"), order: 0)));
        Assert.Throws<ArgumentNullException>("builder", () => PopulateMetadata<Results<ProvidesMetadataResult1, ProvidesMetadataResult2, ProvidesMetadataResult3, ProvidesMetadataResult4>>(((Delegate)GeneratedCodeIsUpToDate).GetMethodInfo(), null));
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    [InlineData(4, typeof(ChecksumResult4))]
    [InlineData(5, typeof(ChecksumResult5))]
    public void ResultsOfTResult1TResult2TResult3TResult4TResult5_Result_IsAssignedResult(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(),
                2 => new ChecksumResult2(),
                3 => new ChecksumResult3(),
                4 => new ChecksumResult4(),
                _ => new ChecksumResult5()
            };
        }

        // Act
        var result = MyApi(input);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
    }

    [Theory]
    [InlineData(1)]
    [InlineData(2)]
    [InlineData(3)]
    [InlineData(4)]
    [InlineData(5)]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5_ExecuteResult_ExecutesAssignedResult(int input)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5> MyApi(int checksum)
        {
            return checksum switch
            {
                1 => new ChecksumResult1(checksum),
                2 => new ChecksumResult2(checksum),
                3 => new ChecksumResult3(checksum),
                4 => new ChecksumResult4(checksum),
                _ => new ChecksumResult5(checksum)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Fact]
    public void ResultsOfTResult1TResult2TResult3TResult4TResult5_Throws_ArgumentNullException_WhenHttpContextIsNull()
    {
        // Arrange
        Results<ChecksumResult1, NoContent> MyApi()
        {
            return new ChecksumResult1(1);
        }
        HttpContext httpContext = null;

        // Act & Assert
        var result = MyApi();

        Assert.ThrowsAsync<ArgumentNullException>(async () =>
        {
            await result.ExecuteAsync(httpContext);
        });
    }

    [Fact]
    public void ResultsOfTResult1TResult2TResult3TResult4TResult5_Throws_InvalidOperationException_WhenResultIsNull()
    {
        // Arrange
        Results<ChecksumResult1, NoContent> MyApi()
        {
            return new ChecksumResult1(1);
        }
        var httpContext = GetHttpContext();

        // Act & Assert
        var result = MyApi();

        Assert.ThrowsAsync<InvalidOperationException>(async () =>
        {
            await result.ExecuteAsync(httpContext);
        });
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    [InlineData(4, typeof(ChecksumResult4))]
    [InlineData(5, typeof(ChecksumResult5))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5_AcceptsIResult_AsFirstTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<IResult, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                2 => new ChecksumResult2(2),
                3 => new ChecksumResult3(3),
                4 => new ChecksumResult4(4),
                _ => new ChecksumResult5(5)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    [InlineData(4, typeof(ChecksumResult4))]
    [InlineData(5, typeof(ChecksumResult5))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5_AcceptsIResult_AsSecondTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, IResult, ChecksumResult3, ChecksumResult4, ChecksumResult5> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                2 => new ChecksumResult2(2),
                3 => new ChecksumResult3(3),
                4 => new ChecksumResult4(4),
                _ => new ChecksumResult5(5)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    [InlineData(4, typeof(ChecksumResult4))]
    [InlineData(5, typeof(ChecksumResult5))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5_AcceptsIResult_AsThirdTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2, IResult, ChecksumResult4, ChecksumResult5> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                2 => new ChecksumResult2(2),
                3 => new ChecksumResult3(3),
                4 => new ChecksumResult4(4),
                _ => new ChecksumResult5(5)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    [InlineData(4, typeof(ChecksumResult4))]
    [InlineData(5, typeof(ChecksumResult5))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5_AcceptsIResult_AsFourthTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, IResult, ChecksumResult5> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                2 => new ChecksumResult2(2),
                3 => new ChecksumResult3(3),
                4 => new ChecksumResult4(4),
                _ => new ChecksumResult5(5)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    [InlineData(4, typeof(ChecksumResult4))]
    [InlineData(5, typeof(ChecksumResult5))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5_AcceptsIResult_AsFifthTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, IResult> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                2 => new ChecksumResult2(2),
                3 => new ChecksumResult3(3),
                4 => new ChecksumResult4(4),
                _ => new ChecksumResult5(5)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(3, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(4, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(5, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(6, typeof(ChecksumResult6))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5_AcceptsNestedResultsOfT_AsFirstTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>, ChecksumResult6> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult2(2),
                3 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult3(3),
                4 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult4(4),
                5 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult5(5),
                _ => new ChecksumResult6(6)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(3, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(4, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(5, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(6, typeof(ChecksumResult6))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5_AcceptsNestedResultsOfT_AsSecondTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>, ChecksumResult6> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult2(2),
                3 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult3(3),
                4 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult4(4),
                5 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult5(5),
                _ => new ChecksumResult6(6)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(3, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(4, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(5, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(6, typeof(ChecksumResult6))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5_AcceptsNestedResultsOfT_AsThirdTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>, ChecksumResult6> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult2(2),
                3 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult3(3),
                4 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult4(4),
                5 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult5(5),
                _ => new ChecksumResult6(6)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(3, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(4, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(5, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(6, typeof(ChecksumResult6))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5_AcceptsNestedResultsOfT_AsFourthTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>, ChecksumResult6> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult2(2),
                3 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult3(3),
                4 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult4(4),
                5 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult5(5),
                _ => new ChecksumResult6(6)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(3, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(4, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(5, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>))]
    [InlineData(6, typeof(ChecksumResult6))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5_AcceptsNestedResultsOfT_AsFifthTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>, ChecksumResult6> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult2(2),
                3 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult3(3),
                4 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult4(4),
                5 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5>)new ChecksumResult5(5),
                _ => new ChecksumResult6(6)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Fact]
    public void ResultsOfTResult1TResult2TResult3TResult4TResult5_PopulateMetadata_PopulatesMetadataFromTypeArgsThatImplementIEndpointMetadataProvider()
    {
        // Arrange
        Results<ProvidesMetadataResult1, ProvidesMetadataResult2, ProvidesMetadataResult3, ProvidesMetadataResult4, ProvidesMetadataResult5> MyApi() { throw new NotImplementedException(); }
        var builder = new RouteEndpointBuilder(requestDelegate: null, RoutePatternFactory.Parse("/"), order: 0);

        // Act
        PopulateMetadata<Results<ProvidesMetadataResult1, ProvidesMetadataResult2, ProvidesMetadataResult3, ProvidesMetadataResult4, ProvidesMetadataResult5>>(((Delegate)MyApi).GetMethodInfo(), builder);

        // Assert
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult1) });
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult2) });
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult3) });
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult4) });
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult5) });
    }

    [Fact]
    public void ResultsOfTResult1TResult2TResult3TResult4TResult5_PopulateMetadata_Throws_ArgumentNullException_WhenMethodOrBuilderAreNull()
    {
        // Act & Assert
        Assert.Throws<ArgumentNullException>("method", () => PopulateMetadata<Results<ProvidesMetadataResult1, ProvidesMetadataResult2, ProvidesMetadataResult3, ProvidesMetadataResult4, ProvidesMetadataResult5>>(null, new RouteEndpointBuilder(requestDelegate: null, RoutePatternFactory.Parse("/"), order: 0)));
        Assert.Throws<ArgumentNullException>("builder", () => PopulateMetadata<Results<ProvidesMetadataResult1, ProvidesMetadataResult2, ProvidesMetadataResult3, ProvidesMetadataResult4, ProvidesMetadataResult5>>(((Delegate)GeneratedCodeIsUpToDate).GetMethodInfo(), null));
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    [InlineData(4, typeof(ChecksumResult4))]
    [InlineData(5, typeof(ChecksumResult5))]
    [InlineData(6, typeof(ChecksumResult6))]
    public void ResultsOfTResult1TResult2TResult3TResult4TResult5TResult6_Result_IsAssignedResult(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(),
                2 => new ChecksumResult2(),
                3 => new ChecksumResult3(),
                4 => new ChecksumResult4(),
                5 => new ChecksumResult5(),
                _ => new ChecksumResult6()
            };
        }

        // Act
        var result = MyApi(input);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
    }

    [Theory]
    [InlineData(1)]
    [InlineData(2)]
    [InlineData(3)]
    [InlineData(4)]
    [InlineData(5)]
    [InlineData(6)]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5TResult6_ExecuteResult_ExecutesAssignedResult(int input)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6> MyApi(int checksum)
        {
            return checksum switch
            {
                1 => new ChecksumResult1(checksum),
                2 => new ChecksumResult2(checksum),
                3 => new ChecksumResult3(checksum),
                4 => new ChecksumResult4(checksum),
                5 => new ChecksumResult5(checksum),
                _ => new ChecksumResult6(checksum)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Fact]
    public void ResultsOfTResult1TResult2TResult3TResult4TResult5TResult6_Throws_ArgumentNullException_WhenHttpContextIsNull()
    {
        // Arrange
        Results<ChecksumResult1, NoContent> MyApi()
        {
            return new ChecksumResult1(1);
        }
        HttpContext httpContext = null;

        // Act & Assert
        var result = MyApi();

        Assert.ThrowsAsync<ArgumentNullException>(async () =>
        {
            await result.ExecuteAsync(httpContext);
        });
    }

    [Fact]
    public void ResultsOfTResult1TResult2TResult3TResult4TResult5TResult6_Throws_InvalidOperationException_WhenResultIsNull()
    {
        // Arrange
        Results<ChecksumResult1, NoContent> MyApi()
        {
            return new ChecksumResult1(1);
        }
        var httpContext = GetHttpContext();

        // Act & Assert
        var result = MyApi();

        Assert.ThrowsAsync<InvalidOperationException>(async () =>
        {
            await result.ExecuteAsync(httpContext);
        });
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    [InlineData(4, typeof(ChecksumResult4))]
    [InlineData(5, typeof(ChecksumResult5))]
    [InlineData(6, typeof(ChecksumResult6))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5TResult6_AcceptsIResult_AsFirstTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<IResult, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                2 => new ChecksumResult2(2),
                3 => new ChecksumResult3(3),
                4 => new ChecksumResult4(4),
                5 => new ChecksumResult5(5),
                _ => new ChecksumResult6(6)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    [InlineData(4, typeof(ChecksumResult4))]
    [InlineData(5, typeof(ChecksumResult5))]
    [InlineData(6, typeof(ChecksumResult6))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5TResult6_AcceptsIResult_AsSecondTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, IResult, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                2 => new ChecksumResult2(2),
                3 => new ChecksumResult3(3),
                4 => new ChecksumResult4(4),
                5 => new ChecksumResult5(5),
                _ => new ChecksumResult6(6)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    [InlineData(4, typeof(ChecksumResult4))]
    [InlineData(5, typeof(ChecksumResult5))]
    [InlineData(6, typeof(ChecksumResult6))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5TResult6_AcceptsIResult_AsThirdTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2, IResult, ChecksumResult4, ChecksumResult5, ChecksumResult6> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                2 => new ChecksumResult2(2),
                3 => new ChecksumResult3(3),
                4 => new ChecksumResult4(4),
                5 => new ChecksumResult5(5),
                _ => new ChecksumResult6(6)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    [InlineData(4, typeof(ChecksumResult4))]
    [InlineData(5, typeof(ChecksumResult5))]
    [InlineData(6, typeof(ChecksumResult6))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5TResult6_AcceptsIResult_AsFourthTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, IResult, ChecksumResult5, ChecksumResult6> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                2 => new ChecksumResult2(2),
                3 => new ChecksumResult3(3),
                4 => new ChecksumResult4(4),
                5 => new ChecksumResult5(5),
                _ => new ChecksumResult6(6)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    [InlineData(4, typeof(ChecksumResult4))]
    [InlineData(5, typeof(ChecksumResult5))]
    [InlineData(6, typeof(ChecksumResult6))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5TResult6_AcceptsIResult_AsFifthTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, IResult, ChecksumResult6> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                2 => new ChecksumResult2(2),
                3 => new ChecksumResult3(3),
                4 => new ChecksumResult4(4),
                5 => new ChecksumResult5(5),
                _ => new ChecksumResult6(6)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(ChecksumResult1))]
    [InlineData(2, typeof(ChecksumResult2))]
    [InlineData(3, typeof(ChecksumResult3))]
    [InlineData(4, typeof(ChecksumResult4))]
    [InlineData(5, typeof(ChecksumResult5))]
    [InlineData(6, typeof(ChecksumResult6))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5TResult6_AcceptsIResult_AsSixthTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, IResult> MyApi(int id)
        {
            return id switch
            {
                1 => new ChecksumResult1(1),
                2 => new ChecksumResult2(2),
                3 => new ChecksumResult3(3),
                4 => new ChecksumResult4(4),
                5 => new ChecksumResult5(5),
                _ => new ChecksumResult6(6)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(3, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(4, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(5, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(6, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(7, typeof(ChecksumResult7))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5TResult6_AcceptsNestedResultsOfT_AsFirstTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>, ChecksumResult7> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult2(2),
                3 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult3(3),
                4 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult4(4),
                5 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult5(5),
                6 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult6(6),
                _ => new ChecksumResult7(7)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(3, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(4, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(5, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(6, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(7, typeof(ChecksumResult7))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5TResult6_AcceptsNestedResultsOfT_AsSecondTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>, ChecksumResult7> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult2(2),
                3 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult3(3),
                4 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult4(4),
                5 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult5(5),
                6 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult6(6),
                _ => new ChecksumResult7(7)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(3, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(4, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(5, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(6, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(7, typeof(ChecksumResult7))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5TResult6_AcceptsNestedResultsOfT_AsThirdTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>, ChecksumResult7> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult2(2),
                3 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult3(3),
                4 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult4(4),
                5 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult5(5),
                6 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult6(6),
                _ => new ChecksumResult7(7)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(3, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(4, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(5, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(6, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(7, typeof(ChecksumResult7))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5TResult6_AcceptsNestedResultsOfT_AsFourthTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>, ChecksumResult7> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult2(2),
                3 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult3(3),
                4 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult4(4),
                5 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult5(5),
                6 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult6(6),
                _ => new ChecksumResult7(7)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(3, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(4, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(5, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(6, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(7, typeof(ChecksumResult7))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5TResult6_AcceptsNestedResultsOfT_AsFifthTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>, ChecksumResult7> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult2(2),
                3 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult3(3),
                4 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult4(4),
                5 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult5(5),
                6 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult6(6),
                _ => new ChecksumResult7(7)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Theory]
    [InlineData(1, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(2, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(3, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(4, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(5, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(6, typeof(Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>))]
    [InlineData(7, typeof(ChecksumResult7))]
    public async Task ResultsOfTResult1TResult2TResult3TResult4TResult5TResult6_AcceptsNestedResultsOfT_AsSixthTypeArg(int input, Type expectedResultType)
    {
        // Arrange
        Results<Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>, ChecksumResult7> MyApi(int id)
        {
            return id switch
            {
                1 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult1(1),
                2 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult2(2),
                3 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult3(3),
                4 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult4(4),
                5 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult5(5),
                6 => (Results<ChecksumResult1, ChecksumResult2, ChecksumResult3, ChecksumResult4, ChecksumResult5, ChecksumResult6>)new ChecksumResult6(6),
                _ => new ChecksumResult7(7)
            };
        }
        var httpContext = GetHttpContext();

        // Act
        var result = MyApi(input);
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.IsType(expectedResultType, result.Result);
        Assert.Equal(input, httpContext.Items[nameof(ChecksumResult.Checksum)]);
    }

    [Fact]
    public void ResultsOfTResult1TResult2TResult3TResult4TResult5TResult6_PopulateMetadata_PopulatesMetadataFromTypeArgsThatImplementIEndpointMetadataProvider()
    {
        // Arrange
        Results<ProvidesMetadataResult1, ProvidesMetadataResult2, ProvidesMetadataResult3, ProvidesMetadataResult4, ProvidesMetadataResult5, ProvidesMetadataResult6> MyApi() { throw new NotImplementedException(); }
        var builder = new RouteEndpointBuilder(requestDelegate: null, RoutePatternFactory.Parse("/"), order: 0);

        // Act
        PopulateMetadata<Results<ProvidesMetadataResult1, ProvidesMetadataResult2, ProvidesMetadataResult3, ProvidesMetadataResult4, ProvidesMetadataResult5, ProvidesMetadataResult6>>(((Delegate)MyApi).GetMethodInfo(), builder);

        // Assert
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult1) });
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult2) });
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult3) });
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult4) });
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult5) });
        Assert.Contains(builder.Metadata, m => m is ResultTypeProvidedMetadata { SourceTypeName: nameof(ProvidesMetadataResult6) });
    }

    [Fact]
    public void ResultsOfTResult1TResult2TResult3TResult4TResult5TResult6_PopulateMetadata_Throws_ArgumentNullException_WhenMethodOrBuilderAreNull()
    {
        // Act & Assert
        Assert.Throws<ArgumentNullException>("method", () => PopulateMetadata<Results<ProvidesMetadataResult1, ProvidesMetadataResult2, ProvidesMetadataResult3, ProvidesMetadataResult4, ProvidesMetadataResult5, ProvidesMetadataResult6>>(null, new RouteEndpointBuilder(requestDelegate: null, RoutePatternFactory.Parse("/"), order: 0)));
        Assert.Throws<ArgumentNullException>("builder", () => PopulateMetadata<Results<ProvidesMetadataResult1, ProvidesMetadataResult2, ProvidesMetadataResult3, ProvidesMetadataResult4, ProvidesMetadataResult5, ProvidesMetadataResult6>>(((Delegate)GeneratedCodeIsUpToDate).GetMethodInfo(), null));
    }

    abstract class ChecksumResult : IResult
    {
        public ChecksumResult(int checksum = 0)
        {
            Checksum = checksum;
        }

        public int Checksum { get; }

        public Task ExecuteAsync(HttpContext httpContext)
        {
            httpContext.Items[nameof(ChecksumResult.Checksum)] = Checksum;
            return Task.CompletedTask;
        }
    }

    class ChecksumResult1 : ChecksumResult
    {
        public ChecksumResult1(int checksum = 0) : base(checksum) { }
    }
    class ProvidesMetadataResult1 : IResult, IEndpointMetadataProvider
    {
        public Task ExecuteAsync(HttpContext httpContext) => Task.CompletedTask;

        public static void PopulateMetadata(MethodInfo method, EndpointBuilder builder)
        {
            builder.Metadata.Add(new ResultTypeProvidedMetadata { SourceTypeName = nameof(ProvidesMetadataResult1) });
        }
    }

    class ChecksumResult2 : ChecksumResult
    {
        public ChecksumResult2(int checksum = 0) : base(checksum) { }
    }
    class ProvidesMetadataResult2 : IResult, IEndpointMetadataProvider
    {
        public Task ExecuteAsync(HttpContext httpContext) => Task.CompletedTask;

        public static void PopulateMetadata(MethodInfo method, EndpointBuilder builder)
        {
            builder.Metadata.Add(new ResultTypeProvidedMetadata { SourceTypeName = nameof(ProvidesMetadataResult2) });
        }
    }

    class ChecksumResult3 : ChecksumResult
    {
        public ChecksumResult3(int checksum = 0) : base(checksum) { }
    }
    class ProvidesMetadataResult3 : IResult, IEndpointMetadataProvider
    {
        public Task ExecuteAsync(HttpContext httpContext) => Task.CompletedTask;

        public static void PopulateMetadata(MethodInfo method, EndpointBuilder builder)
        {
            builder.Metadata.Add(new ResultTypeProvidedMetadata { SourceTypeName = nameof(ProvidesMetadataResult3) });
        }
    }

    class ChecksumResult4 : ChecksumResult
    {
        public ChecksumResult4(int checksum = 0) : base(checksum) { }
    }
    class ProvidesMetadataResult4 : IResult, IEndpointMetadataProvider
    {
        public Task ExecuteAsync(HttpContext httpContext) => Task.CompletedTask;

        public static void PopulateMetadata(MethodInfo method, EndpointBuilder builder)
        {
            builder.Metadata.Add(new ResultTypeProvidedMetadata { SourceTypeName = nameof(ProvidesMetadataResult4) });
        }
    }

    class ChecksumResult5 : ChecksumResult
    {
        public ChecksumResult5(int checksum = 0) : base(checksum) { }
    }
    class ProvidesMetadataResult5 : IResult, IEndpointMetadataProvider
    {
        public Task ExecuteAsync(HttpContext httpContext) => Task.CompletedTask;

        public static void PopulateMetadata(MethodInfo method, EndpointBuilder builder)
        {
            builder.Metadata.Add(new ResultTypeProvidedMetadata { SourceTypeName = nameof(ProvidesMetadataResult5) });
        }
    }

    class ChecksumResult6 : ChecksumResult
    {
        public ChecksumResult6(int checksum = 0) : base(checksum) { }
    }
    class ProvidesMetadataResult6 : IResult, IEndpointMetadataProvider
    {
        public Task ExecuteAsync(HttpContext httpContext) => Task.CompletedTask;

        public static void PopulateMetadata(MethodInfo method, EndpointBuilder builder)
        {
            builder.Metadata.Add(new ResultTypeProvidedMetadata { SourceTypeName = nameof(ProvidesMetadataResult6) });
        }
    }
    class ChecksumResult7 : ChecksumResult
    {
        public ChecksumResult7(int checksum = 0) : base(checksum) { }
    }
    class ProvidesMetadataResult7 : IResult, IEndpointMetadataProvider
    {
        public Task ExecuteAsync(HttpContext httpContext) => Task.CompletedTask;

        public static void PopulateMetadata(MethodInfo method, EndpointBuilder builder)
        {
            builder.Metadata.Add(new ResultTypeProvidedMetadata { SourceTypeName = nameof(ProvidesMetadataResult7) });
        }
    }

}
