#!/usr/bin/python3

'''
Apport package hook for Squid

Copyright (C) 2022 Canonical Ltd.
Author: Bryce Harrington <bryce@canonical.com>

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.  See http://www.gnu.org/copyleft/gpl.html for
the full text of the license.
'''

import os.path
from apport.hookutils import attach_file_if_exists


def add_info(report, ui=None):
    '''Attaches squid-specific information to the Apport bug report.'''
    def _add_file(report, filepath):
        filename = os.path.basename(filepath)
        attach_file_if_exists(report, filepath, key=filename)

    # Configs
    _add_file(report, '/etc/squid/squid.conf')
    _add_file(report, '/etc/squid/squid.d/debian.conf')

    if ui is None:
        return

    # Logs
    response = ui.yesno(
        "The contents of your Squid cache.log and access.log files "
        "may help developers diagnose your bug more quickly.  "
        "However, they may contain sensitive " "information.  "
        "Do you want to include them in your bug report?"
    )
    if response is None:
        # user cancelled
        raise StopIteration
    if response is True:
        # Attach files
        _add_file(report, '/var/log/squid/access.log')
        _add_file(report, '/var/log/squid/cache.log')


### DEBUGGING ###
if __name__ == '__main__':
    report = {}
    add_info(report, None)
    for key in report:
        print(f'[{key}]\n{report[key]}')
