/*
 * Copyright (c) 2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if defined(ARM_COMPUTE_ENABLE_SME2)

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>
#include <limits>

namespace arm_gemm {

void sme2_gemv_fp16_mla_16VL (
    const __fp16 *A_ptr, const __fp16 *B_ptr, __fp16 *output_ptr,
    size_t N, size_t K,
    const __fp16 *bias, Activation act, bool
)
{
    struct KernelArgs {
        __fp16 maxval = static_cast<__fp16>(std::numeric_limits<float>::infinity());
        __fp16 minval = - static_cast<__fp16>(std::numeric_limits<float>::infinity());
        const __fp16 *B_ptr = {};
        size_t output_offset = {};
        unsigned int input_initial_col = {};
    } ka;

    unsigned long flags=0;
    ka.B_ptr = B_ptr;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            ka.maxval = static_cast<__fp16>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            ka.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      "ptrue p8.b\n"
      ".inst 0xd503477f  // SMSTART ZA\n"
      "mov x9, #0x0\n"
      "cnth x28, ALL, MUL #4\n"
      "mov x27, %x[B_ptr]\n"
      "add x26, %x[N], x28\n"
      "mov x25, %x[output_ptr]\n"
      "sub x26, x26, #0x1\n"
      "ptrue p1.b\n"
      "udiv x26, x26, x28\n"
      ".inst 0x25207811  // ptrue pn9.b\n"
      "add x22, x26, #0x3\n"
      "mov x21, #0x1\n"
      "and x22, x22, #0xfffffffffffffffc\n"
      "mul x22, x22, x28\n"
      "mul x22, x22, %x[K]\n"
      "lsl x22, x22, #0x1\n"
      "1:"  // RHS size check loop
      "cmp x22, #0x200000\n"
      "blt 2f\n"
      "tbnz x22, #0, 3f\n"
      "lsr x22, x22, #0x1\n"
      "lsl x21, x21, #0x1\n"
      "b 1b\n"
      "2:"  // RHS do prefetch
      "lsl x20, x22, #0x26\n"
      "sub x21, x21, #0x1\n"
      "lsl x21, x21, #0x16\n"
      "orr x22, x22, x20\n"
      "orr x22, x22, x21\n"
      ".inst 0xf8b64b7a  // rprfm pldonce, x22, [x27]\n"
      "3:"  // RHS prefetch exit
      "mov x24, %x[bias]\n"
      "4:"  // Column loop
      "cmp x26, #0x4\n"
      "bge 28f\n"
      "cmp x26, #0x2\n"
      "bgt 20f\n"
      "beq 12f\n"
      "mov x23, %x[A_ptr]\n"
      "lsl x21, %x[K], #0x1\n"
      "mov x20, %x[N]\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      ".inst 0x257467f0  // whilelt p8.h, XZR, x20, VLx4\n"
      "cbz x24, 5f\n"
      ".inst 0xa040a708  // ld1h { z8.h-z11.h }, pn9.b/Z, [x24]\n"
      ".inst 0xc0042d00  // mova za.d[x9, #0], { z8.d-z11.d }\n"
      "b 6f\n"
      "5:"  // Width 1: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "6:"  // Width 1: setup done
      "cmp x22, #0x8\n"
      "ble 8f\n"
      "7:"  // Width 1: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x22\n"
      ".inst 0xa040a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      "ld1rqh { z3.h }, p0/Z, [x23]\n"
      "sub x22, x22, #0x8\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa040a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      "cmp x22, #0x8\n"
      ".inst 0xa040a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc113b280  // fmla za.h[x9, 0], { z20.h-z23.h }, z3.h[0]\n"
      ".inst 0xa040a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xa040a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xa040a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc113b388  // fmla za.h[x9, 0], { z28.h-z31.h }, z3.h[1]\n"
      ".inst 0xa040a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xa040a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc113b500  // fmla za.h[x9, 0], { z8.h-z11.h }, z3.h[2]\n"
      ".inst 0xc113b488  // fmla za.h[x9, 0], { z4.h-z7.h }, z3.h[3]\n"
      ".inst 0xc113ba80  // fmla za.h[x9, 0], { z20.h-z23.h }, z3.h[4]\n"
      ".inst 0xc113b988  // fmla za.h[x9, 0], { z12.h-z15.h }, z3.h[5]\n"
      ".inst 0xc113bf80  // fmla za.h[x9, 0], { z28.h-z31.h }, z3.h[6]\n"
      ".inst 0xc113be08  // fmla za.h[x9, 0], { z16.h-z19.h }, z3.h[7]\n"
      "bgt 7b\n"
      "8:"  // Width 1: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x22\n"
      ".inst 0xa040a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      "ld1rqh { z15.h }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb280  // fmla za.h[x9, 0], { z20.h-z23.h }, z15.h[0]\n"
      "ble 9f\n"
      ".inst 0xa040a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb208  // fmla za.h[x9, 0], { z16.h-z19.h }, z15.h[1]\n"
      "ble 9f\n"
      ".inst 0xa040a761  // ldnt1h { z0.h-z3.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb400  // fmla za.h[x9, 0], { z0.h-z3.h }, z15.h[2]\n"
      "ble 9f\n"
      ".inst 0xa040a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb788  // fmla za.h[x9, 0], { z28.h-z31.h }, z15.h[3]\n"
      "ble 9f\n"
      ".inst 0xa040a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb880  // fmla za.h[x9, 0], { z4.h-z7.h }, z15.h[4]\n"
      "ble 9f\n"
      ".inst 0xa040a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fba08  // fmla za.h[x9, 0], { z16.h-z19.h }, z15.h[5]\n"
      "ble 9f\n"
      ".inst 0xa040a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fbc80  // fmla za.h[x9, 0], { z4.h-z7.h }, z15.h[6]\n"
      "ble 9f\n"
      ".inst 0xa040a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fbf88  // fmla za.h[x9, 0], { z28.h-z31.h }, z15.h[7]\n"
      "9:"  // Width 1: Multiply loop: multiply skip
      "tbz %x[flags], #1, 10f\n"
      "add x21, %x[args_ptr], %[offset_min]\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      ".inst 0xc0062c08  // mova { z8.d-z11.d }, za.d[x9, #0]\n"
      "ld1rh { z4.h }, p1/Z, [x21]\n"
      "ld1rh { z21.h }, p1/Z, [x20]\n"
      ".inst 0xc175c888  // fclamp { z8.h-z11.h }, z4.h, z21.h\n"
      ".inst 0xa060a328  // st1h { z8.h-z11.h }, p8, [x25]\n"
      "addvl x25, x25, #4\n"
      "b 11f\n"
      "10:"  // Width 1: No activation
      ".inst 0xc0062c00  // mova { z0.d-z3.d }, za.d[x9, #0]\n"
      ".inst 0xa060a320  // st1h { z0.h-z3.h }, p8, [x25]\n"
      "addvl x25, x25, #4\n"
      "11:"  // Width 1: Output done
      "b 36f\n"
      "12:"  // Width 2
      "mov x23, %x[A_ptr]\n"
      "lsl x21, %x[K], #0x1\n"
      "sub x20, %x[N], x28\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      ".inst 0x257467f0  // whilelt p8.h, XZR, x20, VLx4\n"
      "cbz x24, 13f\n"
      ".inst 0xa040a71c  // ld1h { z28.h-z31.h }, pn9.b/Z, [x24]\n"
      ".inst 0xa041a70c  // ld1h { z12.h-z15.h }, pn9.b/Z, [x24, #0x4, MUL VL]\n"
      ".inst 0xc0042f80  // mova za.d[x9, #0], { z28.d-z31.d }\n"
      ".inst 0xc0042d81  // mova za.d[x9, #1], { z12.d-z15.d }\n"
      "b 14f\n"
      "13:"  // Width 2: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "14:"  // Width 2: setup done
      "cmp x22, #0x8\n"
      "ble 16f\n"
      "15:"  // Width 2: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x22\n"
      ".inst 0xa040a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27]\n"
      "sub x22, x22, #0x8\n"
      "ld1rqh { z3.h }, p0/Z, [x23]\n"
      "cmp x22, #0x8\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa041a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xa040a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc113b100  // fmla za.h[x9, 0], { z8.h-z11.h }, z3.h[0]\n"
      ".inst 0xa041a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc113b301  // fmla za.h[x9, 1], { z24.h-z27.h }, z3.h[0]\n"
      ".inst 0xa040a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27]\n"
      ".inst 0xa041a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc113b288  // fmla za.h[x9, 0], { z20.h-z23.h }, z3.h[1]\n"
      ".inst 0xa040a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc113b209  // fmla za.h[x9, 1], { z16.h-z19.h }, z3.h[1]\n"
      ".inst 0xa041a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xa040a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc113b500  // fmla za.h[x9, 0], { z8.h-z11.h }, z3.h[2]\n"
      ".inst 0xa041a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc113b581  // fmla za.h[x9, 1], { z12.h-z15.h }, z3.h[2]\n"
      ".inst 0xa040a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27]\n"
      ".inst 0xa041a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc113b688  // fmla za.h[x9, 0], { z20.h-z23.h }, z3.h[3]\n"
      ".inst 0xa040a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc113b709  // fmla za.h[x9, 1], { z24.h-z27.h }, z3.h[3]\n"
      ".inst 0xa041a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xa040a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc113bb80  // fmla za.h[x9, 0], { z28.h-z31.h }, z3.h[4]\n"
      ".inst 0xa041a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc113b881  // fmla za.h[x9, 1], { z4.h-z7.h }, z3.h[4]\n"
      ".inst 0xc113b988  // fmla za.h[x9, 0], { z12.h-z15.h }, z3.h[5]\n"
      ".inst 0xc113ba09  // fmla za.h[x9, 1], { z16.h-z19.h }, z3.h[5]\n"
      ".inst 0xc113bd00  // fmla za.h[x9, 0], { z8.h-z11.h }, z3.h[6]\n"
      ".inst 0xc113be81  // fmla za.h[x9, 1], { z20.h-z23.h }, z3.h[6]\n"
      ".inst 0xc113bf08  // fmla za.h[x9, 0], { z24.h-z27.h }, z3.h[7]\n"
      ".inst 0xc113bf89  // fmla za.h[x9, 1], { z28.h-z31.h }, z3.h[7]\n"
      "bgt 15b\n"
      "16:"  // Width 2: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x22\n"
      ".inst 0xa040a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      "ld1rqh { z15.h }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa041a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb380  // fmla za.h[x9, 0], { z28.h-z31.h }, z15.h[0]\n"
      ".inst 0xc11fb101  // fmla za.h[x9, 1], { z8.h-z11.h }, z15.h[0]\n"
      "ble 17f\n"
      ".inst 0xa040a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      ".inst 0xa041a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb088  // fmla za.h[x9, 0], { z4.h-z7.h }, z15.h[1]\n"
      ".inst 0xc11fb289  // fmla za.h[x9, 1], { z20.h-z23.h }, z15.h[1]\n"
      "ble 17f\n"
      ".inst 0xa040a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      ".inst 0xa041a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb600  // fmla za.h[x9, 0], { z16.h-z19.h }, z15.h[2]\n"
      ".inst 0xc11fb501  // fmla za.h[x9, 1], { z8.h-z11.h }, z15.h[2]\n"
      "ble 17f\n"
      ".inst 0xa040a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      ".inst 0xa041a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb508  // fmla za.h[x9, 0], { z8.h-z11.h }, z15.h[3]\n"
      ".inst 0xc11fb789  // fmla za.h[x9, 1], { z28.h-z31.h }, z15.h[3]\n"
      "ble 17f\n"
      ".inst 0xa040a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      ".inst 0xa041a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb880  // fmla za.h[x9, 0], { z4.h-z7.h }, z15.h[4]\n"
      ".inst 0xc11fba01  // fmla za.h[x9, 1], { z16.h-z19.h }, z15.h[4]\n"
      "ble 17f\n"
      ".inst 0xa040a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      ".inst 0xa041a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fbb08  // fmla za.h[x9, 0], { z24.h-z27.h }, z15.h[5]\n"
      ".inst 0xc11fb909  // fmla za.h[x9, 1], { z8.h-z11.h }, z15.h[5]\n"
      "ble 17f\n"
      ".inst 0xa040a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      ".inst 0xa041a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fbf80  // fmla za.h[x9, 0], { z28.h-z31.h }, z15.h[6]\n"
      ".inst 0xc11fbd01  // fmla za.h[x9, 1], { z8.h-z11.h }, z15.h[6]\n"
      "ble 17f\n"
      ".inst 0xa040a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27]\n"
      ".inst 0xa041a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fbf08  // fmla za.h[x9, 0], { z24.h-z27.h }, z15.h[7]\n"
      ".inst 0xc11fbf89  // fmla za.h[x9, 1], { z28.h-z31.h }, z15.h[7]\n"
      "17:"  // Width 2: Multiply loop: multiply skip
      "tbz %x[flags], #1, 18f\n"
      "add x21, %x[args_ptr], %[offset_min]\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      ".inst 0xc0062c10  // mova { z16.d-z19.d }, za.d[x9, #0]\n"
      ".inst 0xc0062c3c  // mova { z28.d-z31.d }, za.d[x9, #1]\n"
      "ld1rh { z15.h }, p1/Z, [x21]\n"
      "ld1rh { z27.h }, p1/Z, [x20]\n"
      ".inst 0xc17bc9f0  // fclamp { z16.h-z19.h }, z15.h, z27.h\n"
      ".inst 0xc17bc9fc  // fclamp { z28.h-z31.h }, z15.h, z27.h\n"
      ".inst 0xa060a730  // st1h { z16.h-z19.h }, pn9.b, [x25]\n"
      ".inst 0xa061a33c  // st1h { z28.h-z31.h }, p8, [x25, #0x4, MUL VL]\n"
      "addvl x25, x25, #8\n"
      "b 19f\n"
      "18:"  // Width 2: No activation
      ".inst 0xc0062c0c  // mova { z12.d-z15.d }, za.d[x9, #0]\n"
      ".inst 0xc0062c3c  // mova { z28.d-z31.d }, za.d[x9, #1]\n"
      ".inst 0xa060a72c  // st1h { z12.h-z15.h }, pn9.b, [x25]\n"
      ".inst 0xa061a33c  // st1h { z28.h-z31.h }, p8, [x25, #0x4, MUL VL]\n"
      "addvl x25, x25, #8\n"
      "19:"  // Width 2: Output done
      "b 36f\n"
      "20:"  // Width 3
      "mov x20, #0x2\n"
      "mov x23, %x[A_ptr]\n"
      "lsl x21, %x[K], #0x1\n"
      "msub x20, x28, x20, %x[N]\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      ".inst 0x257467f0  // whilelt p8.h, XZR, x20, VLx4\n"
      "cbz x24, 21f\n"
      ".inst 0xa040a700  // ld1h { z0.h-z3.h }, pn9.b/Z, [x24]\n"
      ".inst 0xa041a710  // ld1h { z16.h-z19.h }, pn9.b/Z, [x24, #0x4, MUL VL]\n"
      ".inst 0xa042a708  // ld1h { z8.h-z11.h }, pn9.b/Z, [x24, #0x8, MUL VL]\n"
      ".inst 0xc0042c00  // mova za.d[x9, #0], { z0.d-z3.d }\n"
      ".inst 0xc0042e01  // mova za.d[x9, #1], { z16.d-z19.d }\n"
      ".inst 0xc0042d02  // mova za.d[x9, #2], { z8.d-z11.d }\n"
      "b 22f\n"
      "21:"  // Width 3: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "22:"  // Width 3: setup done
      "cmp x22, #0x8\n"
      "ble 24f\n"
      "23:"  // Width 3: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x22\n"
      ".inst 0xa040a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27]\n"
      "sub x22, x22, #0x8\n"
      "ld1rqh { z0.h }, p0/Z, [x23]\n"
      "cmp x22, #0x8\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa041a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc110b180  // fmla za.h[x9, 0], { z12.h-z15.h }, z0.h[0]\n"
      ".inst 0xa040a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc110b281  // fmla za.h[x9, 1], { z20.h-z23.h }, z0.h[0]\n"
      ".inst 0xa041a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc110b102  // fmla za.h[x9, 2], { z8.h-z11.h }, z0.h[0]\n"
      ".inst 0xa042a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc110b088  // fmla za.h[x9, 0], { z4.h-z7.h }, z0.h[1]\n"
      ".inst 0xa040a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc110b309  // fmla za.h[x9, 1], { z24.h-z27.h }, z0.h[1]\n"
      ".inst 0xa041a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc110b18a  // fmla za.h[x9, 2], { z12.h-z15.h }, z0.h[1]\n"
      ".inst 0xa042a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc110b680  // fmla za.h[x9, 0], { z20.h-z23.h }, z0.h[2]\n"
      ".inst 0xa040a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc110b781  // fmla za.h[x9, 1], { z28.h-z31.h }, z0.h[2]\n"
      ".inst 0xa041a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc110b502  // fmla za.h[x9, 2], { z8.h-z11.h }, z0.h[2]\n"
      ".inst 0xa042a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc110b608  // fmla za.h[x9, 0], { z16.h-z19.h }, z0.h[3]\n"
      ".inst 0xa040a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc110b709  // fmla za.h[x9, 1], { z24.h-z27.h }, z0.h[3]\n"
      ".inst 0xa041a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc110b48a  // fmla za.h[x9, 2], { z4.h-z7.h }, z0.h[3]\n"
      ".inst 0xa042a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc110bb80  // fmla za.h[x9, 0], { z28.h-z31.h }, z0.h[4]\n"
      ".inst 0xa040a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc110b901  // fmla za.h[x9, 1], { z8.h-z11.h }, z0.h[4]\n"
      ".inst 0xa041a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc110b882  // fmla za.h[x9, 2], { z4.h-z7.h }, z0.h[4]\n"
      ".inst 0xa042a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc110b988  // fmla za.h[x9, 0], { z12.h-z15.h }, z0.h[5]\n"
      ".inst 0xa040a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc110b909  // fmla za.h[x9, 1], { z8.h-z11.h }, z0.h[5]\n"
      ".inst 0xa041a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc110bb0a  // fmla za.h[x9, 2], { z24.h-z27.h }, z0.h[5]\n"
      ".inst 0xa042a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc110be80  // fmla za.h[x9, 0], { z20.h-z23.h }, z0.h[6]\n"
      ".inst 0xa040a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc110be01  // fmla za.h[x9, 1], { z16.h-z19.h }, z0.h[6]\n"
      ".inst 0xa041a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc110bd82  // fmla za.h[x9, 2], { z12.h-z15.h }, z0.h[6]\n"
      ".inst 0xa042a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc110bc88  // fmla za.h[x9, 0], { z4.h-z7.h }, z0.h[7]\n"
      ".inst 0xc110be09  // fmla za.h[x9, 1], { z16.h-z19.h }, z0.h[7]\n"
      ".inst 0xc110be8a  // fmla za.h[x9, 2], { z20.h-z23.h }, z0.h[7]\n"
      "bgt 23b\n"
      "24:"  // Width 3: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x22\n"
      ".inst 0xa040a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      "ld1rqh { z15.h }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa041a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb080  // fmla za.h[x9, 0], { z4.h-z7.h }, z15.h[0]\n"
      ".inst 0xc11fb301  // fmla za.h[x9, 1], { z24.h-z27.h }, z15.h[0]\n"
      ".inst 0xc11fb382  // fmla za.h[x9, 2], { z28.h-z31.h }, z15.h[0]\n"
      "ble 25f\n"
      ".inst 0xa040a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      ".inst 0xa041a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb308  // fmla za.h[x9, 0], { z24.h-z27.h }, z15.h[1]\n"
      ".inst 0xc11fb089  // fmla za.h[x9, 1], { z4.h-z7.h }, z15.h[1]\n"
      ".inst 0xc11fb10a  // fmla za.h[x9, 2], { z8.h-z11.h }, z15.h[1]\n"
      "ble 25f\n"
      ".inst 0xa040a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      ".inst 0xa041a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb780  // fmla za.h[x9, 0], { z28.h-z31.h }, z15.h[2]\n"
      ".inst 0xc11fb601  // fmla za.h[x9, 1], { z16.h-z19.h }, z15.h[2]\n"
      ".inst 0xc11fb502  // fmla za.h[x9, 2], { z8.h-z11.h }, z15.h[2]\n"
      "ble 25f\n"
      ".inst 0xa040a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      ".inst 0xa041a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb488  // fmla za.h[x9, 0], { z4.h-z7.h }, z15.h[3]\n"
      ".inst 0xc11fb709  // fmla za.h[x9, 1], { z24.h-z27.h }, z15.h[3]\n"
      ".inst 0xc11fb60a  // fmla za.h[x9, 2], { z16.h-z19.h }, z15.h[3]\n"
      "ble 25f\n"
      ".inst 0xa040a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      ".inst 0xa041a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb900  // fmla za.h[x9, 0], { z8.h-z11.h }, z15.h[4]\n"
      ".inst 0xc11fbb01  // fmla za.h[x9, 1], { z24.h-z27.h }, z15.h[4]\n"
      ".inst 0xc11fba02  // fmla za.h[x9, 2], { z16.h-z19.h }, z15.h[4]\n"
      "ble 25f\n"
      ".inst 0xa040a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      ".inst 0xa041a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb908  // fmla za.h[x9, 0], { z8.h-z11.h }, z15.h[5]\n"
      ".inst 0xc11fba09  // fmla za.h[x9, 1], { z16.h-z19.h }, z15.h[5]\n"
      ".inst 0xc11fbb0a  // fmla za.h[x9, 2], { z24.h-z27.h }, z15.h[5]\n"
      "ble 25f\n"
      ".inst 0xa040a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      ".inst 0xa041a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fbd00  // fmla za.h[x9, 0], { z8.h-z11.h }, z15.h[6]\n"
      ".inst 0xc11fbe81  // fmla za.h[x9, 1], { z20.h-z23.h }, z15.h[6]\n"
      ".inst 0xc11fbf02  // fmla za.h[x9, 2], { z24.h-z27.h }, z15.h[6]\n"
      "ble 25f\n"
      ".inst 0xa040a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27]\n"
      ".inst 0xa041a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fbf88  // fmla za.h[x9, 0], { z28.h-z31.h }, z15.h[7]\n"
      ".inst 0xc11fbd09  // fmla za.h[x9, 1], { z8.h-z11.h }, z15.h[7]\n"
      ".inst 0xc11fbe8a  // fmla za.h[x9, 2], { z20.h-z23.h }, z15.h[7]\n"
      "25:"  // Width 3: Multiply loop: multiply skip
      "tbz %x[flags], #1, 26f\n"
      "add x21, %x[args_ptr], %[offset_min]\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      ".inst 0xc0062c0c  // mova { z12.d-z15.d }, za.d[x9, #0]\n"
      ".inst 0xc0062c20  // mova { z0.d-z3.d }, za.d[x9, #1]\n"
      "ld1rh { z17.h }, p1/Z, [x21]\n"
      ".inst 0xc0062c44  // mova { z4.d-z7.d }, za.d[x9, #2]\n"
      "ld1rh { z16.h }, p1/Z, [x20]\n"
      ".inst 0xc170ca2c  // fclamp { z12.h-z15.h }, z17.h, z16.h\n"
      ".inst 0xc170ca20  // fclamp { z0.h-z3.h }, z17.h, z16.h\n"
      ".inst 0xa060a72c  // st1h { z12.h-z15.h }, pn9.b, [x25]\n"
      ".inst 0xc170ca24  // fclamp { z4.h-z7.h }, z17.h, z16.h\n"
      ".inst 0xa061a720  // st1h { z0.h-z3.h }, pn9.b, [x25, #0x4, MUL VL]\n"
      ".inst 0xa062a324  // st1h { z4.h-z7.h }, p8, [x25, #0x8, MUL VL]\n"
      "addvl x25, x25, #12\n"
      "b 27f\n"
      "26:"  // Width 3: No activation
      ".inst 0xc0062c10  // mova { z16.d-z19.d }, za.d[x9, #0]\n"
      ".inst 0xc0062c38  // mova { z24.d-z27.d }, za.d[x9, #1]\n"
      ".inst 0xc0062c54  // mova { z20.d-z23.d }, za.d[x9, #2]\n"
      ".inst 0xa060a730  // st1h { z16.h-z19.h }, pn9.b, [x25]\n"
      ".inst 0xa061a738  // st1h { z24.h-z27.h }, pn9.b, [x25, #0x4, MUL VL]\n"
      ".inst 0xa062a334  // st1h { z20.h-z23.h }, p8, [x25, #0x8, MUL VL]\n"
      "addvl x25, x25, #12\n"
      "27:"  // Width 3: Output done
      "b 36f\n"
      "28:"  // Width 4
      "mov x20, #0x3\n"
      "mov x23, %x[A_ptr]\n"
      "lsl x21, %x[K], #0x1\n"
      "msub x20, x28, x20, %x[N]\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      ".inst 0x257467f0  // whilelt p8.h, XZR, x20, VLx4\n"
      "cbz x24, 29f\n"
      ".inst 0xa040a714  // ld1h { z20.h-z23.h }, pn9.b/Z, [x24]\n"
      ".inst 0xa041a704  // ld1h { z4.h-z7.h }, pn9.b/Z, [x24, #0x4, MUL VL]\n"
      ".inst 0xa042a708  // ld1h { z8.h-z11.h }, pn9.b/Z, [x24, #0x8, MUL VL]\n"
      ".inst 0xa043a700  // ld1h { z0.h-z3.h }, pn9.b/Z, [x24, #0xc, MUL VL]\n"
      ".inst 0xc0042e80  // mova za.d[x9, #0], { z20.d-z23.d }\n"
      "addvl x24, x24, #16\n"
      ".inst 0xc0042c81  // mova za.d[x9, #1], { z4.d-z7.d }\n"
      ".inst 0xc0042d02  // mova za.d[x9, #2], { z8.d-z11.d }\n"
      ".inst 0xc0042c03  // mova za.d[x9, #3], { z0.d-z3.d }\n"
      "b 30f\n"
      "29:"  // Width 4: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "30:"  // Width 4: setup done
      "cmp x22, #0x8\n"
      "ble 32f\n"
      "31:"  // Width 4: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x22\n"
      ".inst 0xa040a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27]\n"
      "sub x22, x22, #0x8\n"
      "ld1rqh { z4.h }, p0/Z, [x23]\n"
      "cmp x22, #0x8\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa041a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc114b300  // fmla za.h[x9, 0], { z24.h-z27.h }, z4.h[0]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc114b181  // fmla za.h[x9, 1], { z12.h-z15.h }, z4.h[0]\n"
      ".inst 0xa040a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc114b382  // fmla za.h[x9, 2], { z28.h-z31.h }, z4.h[0]\n"
      ".inst 0xa041a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc114b103  // fmla za.h[x9, 3], { z8.h-z11.h }, z4.h[0]\n"
      ".inst 0xa042a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc114b208  // fmla za.h[x9, 0], { z16.h-z19.h }, z4.h[1]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc114b189  // fmla za.h[x9, 1], { z12.h-z15.h }, z4.h[1]\n"
      ".inst 0xa040a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc114b30a  // fmla za.h[x9, 2], { z24.h-z27.h }, z4.h[1]\n"
      ".inst 0xa041a761  // ldnt1h { z0.h-z3.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc114b10b  // fmla za.h[x9, 3], { z8.h-z11.h }, z4.h[1]\n"
      ".inst 0xa042a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc114b580  // fmla za.h[x9, 0], { z12.h-z15.h }, z4.h[2]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc114b401  // fmla za.h[x9, 1], { z0.h-z3.h }, z4.h[2]\n"
      ".inst 0xa040a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc114b702  // fmla za.h[x9, 2], { z24.h-z27.h }, z4.h[2]\n"
      ".inst 0xa041a761  // ldnt1h { z0.h-z3.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc114b503  // fmla za.h[x9, 3], { z8.h-z11.h }, z4.h[2]\n"
      ".inst 0xa042a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc114b788  // fmla za.h[x9, 0], { z28.h-z31.h }, z4.h[3]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc114b409  // fmla za.h[x9, 1], { z0.h-z3.h }, z4.h[3]\n"
      ".inst 0xa040a761  // ldnt1h { z0.h-z3.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc114b50a  // fmla za.h[x9, 2], { z8.h-z11.h }, z4.h[3]\n"
      ".inst 0xa041a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc114b58b  // fmla za.h[x9, 3], { z12.h-z15.h }, z4.h[3]\n"
      ".inst 0xa042a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc114b800  // fmla za.h[x9, 0], { z0.h-z3.h }, z4.h[4]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc114bb01  // fmla za.h[x9, 1], { z24.h-z27.h }, z4.h[4]\n"
      ".inst 0xa040a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc114b982  // fmla za.h[x9, 2], { z12.h-z15.h }, z4.h[4]\n"
      ".inst 0xa041a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc114b903  // fmla za.h[x9, 3], { z8.h-z11.h }, z4.h[4]\n"
      ".inst 0xa042a761  // ldnt1h { z0.h-z3.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc114ba88  // fmla za.h[x9, 0], { z20.h-z23.h }, z4.h[5]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc114ba09  // fmla za.h[x9, 1], { z16.h-z19.h }, z4.h[5]\n"
      ".inst 0xa040a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc114b80a  // fmla za.h[x9, 2], { z0.h-z3.h }, z4.h[5]\n"
      ".inst 0xa041a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc114b98b  // fmla za.h[x9, 3], { z12.h-z15.h }, z4.h[5]\n"
      ".inst 0xa042a761  // ldnt1h { z0.h-z3.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc114bf00  // fmla za.h[x9, 0], { z24.h-z27.h }, z4.h[6]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc114be01  // fmla za.h[x9, 1], { z16.h-z19.h }, z4.h[6]\n"
      ".inst 0xa040a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc114bc02  // fmla za.h[x9, 2], { z0.h-z3.h }, z4.h[6]\n"
      ".inst 0xa041a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc114bd03  // fmla za.h[x9, 3], { z8.h-z11.h }, z4.h[6]\n"
      ".inst 0xa042a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc114be08  // fmla za.h[x9, 0], { z16.h-z19.h }, z4.h[7]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc114bd89  // fmla za.h[x9, 1], { z12.h-z15.h }, z4.h[7]\n"
      ".inst 0xc114be8a  // fmla za.h[x9, 2], { z20.h-z23.h }, z4.h[7]\n"
      ".inst 0xc114bd0b  // fmla za.h[x9, 3], { z8.h-z11.h }, z4.h[7]\n"
      "bgt 31b\n"
      "32:"  // Width 4: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x22\n"
      ".inst 0xa040a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      "ld1rqh { z15.h }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa041a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a761  // ldnt1h { z0.h-z3.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc11fb080  // fmla za.h[x9, 0], { z4.h-z7.h }, z15.h[0]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb201  // fmla za.h[x9, 1], { z16.h-z19.h }, z15.h[0]\n"
      ".inst 0xc11fb102  // fmla za.h[x9, 2], { z8.h-z11.h }, z15.h[0]\n"
      ".inst 0xc11fb003  // fmla za.h[x9, 3], { z0.h-z3.h }, z15.h[0]\n"
      "ble 33f\n"
      ".inst 0xa040a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      ".inst 0xa041a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc11fb088  // fmla za.h[x9, 0], { z4.h-z7.h }, z15.h[1]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb389  // fmla za.h[x9, 1], { z28.h-z31.h }, z15.h[1]\n"
      ".inst 0xc11fb10a  // fmla za.h[x9, 2], { z8.h-z11.h }, z15.h[1]\n"
      ".inst 0xc11fb20b  // fmla za.h[x9, 3], { z16.h-z19.h }, z15.h[1]\n"
      "ble 33f\n"
      ".inst 0xa040a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      ".inst 0xa041a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc11fb480  // fmla za.h[x9, 0], { z4.h-z7.h }, z15.h[2]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb501  // fmla za.h[x9, 1], { z8.h-z11.h }, z15.h[2]\n"
      ".inst 0xc11fb682  // fmla za.h[x9, 2], { z20.h-z23.h }, z15.h[2]\n"
      ".inst 0xc11fb603  // fmla za.h[x9, 3], { z16.h-z19.h }, z15.h[2]\n"
      "ble 33f\n"
      ".inst 0xa040a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      ".inst 0xa041a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a761  // ldnt1h { z0.h-z3.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc11fb788  // fmla za.h[x9, 0], { z28.h-z31.h }, z15.h[3]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fb509  // fmla za.h[x9, 1], { z8.h-z11.h }, z15.h[3]\n"
      ".inst 0xc11fb48a  // fmla za.h[x9, 2], { z4.h-z7.h }, z15.h[3]\n"
      ".inst 0xc11fb40b  // fmla za.h[x9, 3], { z0.h-z3.h }, z15.h[3]\n"
      "ble 33f\n"
      ".inst 0xa040a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      ".inst 0xa041a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc11fbb00  // fmla za.h[x9, 0], { z24.h-z27.h }, z15.h[4]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fba81  // fmla za.h[x9, 1], { z20.h-z23.h }, z15.h[4]\n"
      ".inst 0xc11fba02  // fmla za.h[x9, 2], { z16.h-z19.h }, z15.h[4]\n"
      ".inst 0xc11fb883  // fmla za.h[x9, 3], { z4.h-z7.h }, z15.h[4]\n"
      "ble 33f\n"
      ".inst 0xa040a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      ".inst 0xa041a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc11fba08  // fmla za.h[x9, 0], { z16.h-z19.h }, z15.h[5]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fbb89  // fmla za.h[x9, 1], { z28.h-z31.h }, z15.h[5]\n"
      ".inst 0xc11fba8a  // fmla za.h[x9, 2], { z20.h-z23.h }, z15.h[5]\n"
      ".inst 0xc11fb88b  // fmla za.h[x9, 3], { z4.h-z7.h }, z15.h[5]\n"
      "ble 33f\n"
      ".inst 0xa040a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x1\n"
      ".inst 0xa041a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc11fbc80  // fmla za.h[x9, 0], { z4.h-z7.h }, z15.h[6]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fbd01  // fmla za.h[x9, 1], { z8.h-z11.h }, z15.h[6]\n"
      ".inst 0xc11fbe82  // fmla za.h[x9, 2], { z20.h-z23.h }, z15.h[6]\n"
      ".inst 0xc11fbe03  // fmla za.h[x9, 3], { z16.h-z19.h }, z15.h[6]\n"
      "ble 33f\n"
      ".inst 0xa040a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27]\n"
      ".inst 0xa041a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc11fbf08  // fmla za.h[x9, 0], { z24.h-z27.h }, z15.h[7]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc11fbf89  // fmla za.h[x9, 1], { z28.h-z31.h }, z15.h[7]\n"
      ".inst 0xc11fbc8a  // fmla za.h[x9, 2], { z4.h-z7.h }, z15.h[7]\n"
      ".inst 0xc11fbe8b  // fmla za.h[x9, 3], { z20.h-z23.h }, z15.h[7]\n"
      "33:"  // Width 4: Multiply loop: multiply skip
      "tbz %x[flags], #1, 34f\n"
      "add x21, %x[args_ptr], %[offset_min]\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      ".inst 0xc0062c04  // mova { z4.d-z7.d }, za.d[x9, #0]\n"
      ".inst 0xc0062c2c  // mova { z12.d-z15.d }, za.d[x9, #1]\n"
      "ld1rh { z17.h }, p1/Z, [x21]\n"
      ".inst 0xc0062c40  // mova { z0.d-z3.d }, za.d[x9, #2]\n"
      "ld1rh { z16.h }, p1/Z, [x20]\n"
      ".inst 0xc0062c68  // mova { z8.d-z11.d }, za.d[x9, #3]\n"
      ".inst 0xc170ca24  // fclamp { z4.h-z7.h }, z17.h, z16.h\n"
      ".inst 0xc170ca2c  // fclamp { z12.h-z15.h }, z17.h, z16.h\n"
      ".inst 0xc170ca20  // fclamp { z0.h-z3.h }, z17.h, z16.h\n"
      ".inst 0xa060a724  // st1h { z4.h-z7.h }, pn9.b, [x25]\n"
      ".inst 0xa061a72c  // st1h { z12.h-z15.h }, pn9.b, [x25, #0x4, MUL VL]\n"
      ".inst 0xc170ca28  // fclamp { z8.h-z11.h }, z17.h, z16.h\n"
      ".inst 0xa062a720  // st1h { z0.h-z3.h }, pn9.b, [x25, #0x8, MUL VL]\n"
      ".inst 0xa063a328  // st1h { z8.h-z11.h }, p8, [x25, #0xc, MUL VL]\n"
      "addvl x25, x25, #16\n"
      "b 35f\n"
      "34:"  // Width 4: No activation
      ".inst 0xc0062c08  // mova { z8.d-z11.d }, za.d[x9, #0]\n"
      ".inst 0xc0062c2c  // mova { z12.d-z15.d }, za.d[x9, #1]\n"
      ".inst 0xc0062c50  // mova { z16.d-z19.d }, za.d[x9, #2]\n"
      ".inst 0xc0062c60  // mova { z0.d-z3.d }, za.d[x9, #3]\n"
      ".inst 0xa060a728  // st1h { z8.h-z11.h }, pn9.b, [x25]\n"
      ".inst 0xa061a72c  // st1h { z12.h-z15.h }, pn9.b, [x25, #0x4, MUL VL]\n"
      ".inst 0xa062a730  // st1h { z16.h-z19.h }, pn9.b, [x25, #0x8, MUL VL]\n"
      ".inst 0xa063a320  // st1h { z0.h-z3.h }, p8, [x25, #0xc, MUL VL]\n"
      "addvl x25, x25, #16\n"
      "35:"  // Width 4: Output done
      "subs x26, x26, #0x4\n"
      "sub %x[N], %x[N], x28, LSL #2\n"
      "bgt 4b\n"
      "36:"  // Exit
      ".inst 0xd503467f  // SMSTOP\n"
      "ptrue p8.b\n"
      : [N] "+&r" (N)
      : [A_ptr] "r" (A_ptr), [B_ptr] "r" (B_ptr), [K] "r" (K), [args_ptr] "r" (&ka), [bias] "r" (bias), [flags] "r" (flags), [offset_max] "I" (offsetof(KernelArgs, maxval)), [offset_min] "I" (offsetof(KernelArgs, minval)), [output_ptr] "r" (output_ptr)
      : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "p8", "p9", "p10", "p11", "p12", "p13", "p14", "p15", "x9", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm

#endif  // defined(ARM_COMPUTE_ENABLE_SME2)
