/*
 * $Id: glue-dist.c,v 1.12 2009-05-18 09:03:42 vrsieh Exp $
 *
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <errno.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "glue-dist.h"
#include "glue-io.h"
#include "glue-main.h"

static char def_user[1024];
static char def_host[1024];
static char def_path[1024];

static struct {
	char *name;
	char *user;
	char *host;
	char *path;
	pid_t pid;
	int io;
	int err;
	unsigned int port;
	int fd;
} dist_node[64];
static unsigned int dist_node_count = 0;
static unsigned int myself;

static void
dist_err(int fd, void *_cpssp)
{
	char buffer[256];
	int len;
	int ret;

	for (;;) {
		len = read(fd, buffer, sizeof(buffer));
		if (len <= 0) {
			break;
		}
		ret = write(2, buffer, len);
		assert(ret == len);
	}
}

int
dist_node_lookup(const char *name)
{
	unsigned int i;

	for (i = 0; ; i++) {
		if (i == dist_node_count) {
			/* Not found. */
			return -1;
		}
		if (strcmp(dist_node[i].name, name) == 0) {
			/* Found. */
			return i;
		}
	}
}

int
dist_node_create(
	const char *name,
	const char *user,
	const char *host,
	const char *path
)
{
	unsigned int i;
	char buf[32];

	assert(name);
	if (! user) user = def_user;
	if (! host || strcmp(host, "localhost") == 0) host = def_host;
	if (! path) path = def_path;

	if (dist_node_count == 0
	 && (strcmp(def_user, user) != 0
	  || strcmp(def_host, host) != 0)) {
		/* Configuration error. */
		fprintf(stderr, "%s: ERROR: First configured host must be local host!\n",
				progname);
		return -1;
	}

	for (i = 0; i < dist_node_count; i++) {
		if (strcmp(dist_node[i].name, name) == 0) {
			/* Node with same name found. */
			return -1;
		}
		if (strcmp(dist_node[i].host, host) == 0
		 && strcmp(dist_node[i].path, path) == 0) {
			/* Node with same host/path setting found */
			return -1;
		}
	}

	if (dist_node_count == 0) {
		/*
		 * Use this process.
		 */

	} else {
		/*
		 * Create new process.
		 */
		int io[2];
		int err[2];
		int ret;

		ret = socketpair(PF_UNIX, SOCK_STREAM, 0, io);
		assert(0 <= ret);
		ret = socketpair(PF_UNIX, SOCK_STREAM, 0, err);
		assert(0 <= ret);

		switch (dist_node[i].pid = fork()) {
		case -1:
			/* Error */
			return -1;
		case 0:
			/* Child */
			ret = close(0);
			assert(0 <= ret);
			ret = dup(io[0]);
			assert(ret == 0);

			ret = close(1);
			assert(0 <= ret);
			ret = dup(io[0]);
			assert(ret == 1);

			ret = close(2);
			assert(0 <= ret);
			ret = dup(err[0]);
			assert(ret == 2);

			ret = close(io[0]);
			assert(0 <= ret);
			ret = close(io[1]);
			assert(0 <= ret);

			ret = close(err[0]);
			assert(0 <= ret);
			ret = close(err[1]);
			assert(0 <= ret);

			if (strcmp(def_user, user) != 0
			 || strcmp(def_host, host) != 0) {
				/*
				 * Start with different user name
				 * or
				 * start on remote host.
				 */
				/* Start program via ssh... */
				char cmd[3*1024];

				sprintf(cmd, "env PATH=%s %s -s -B %s",
					getenv("PATH"), progname, path);
				execlp("ssh", "ssh", "-l", user, host,
					cmd, (char *) NULL);
				fprintf(stderr, "%s: ERROR: \"ssh -l %s %s %s -s -B %s\" failed: %s.\n",
						progname,
						user, host, progname, path,
						strerror(errno));
			} else {
				/*
				 * Start with same user name
				 * and
				 * start on localhost.
				 */
				/* Start program directly... */
				execlp(progname,
					progname, "-s", "-B", path,
					(char *) NULL);
				fprintf(stderr, "%s: ERROR: \"%s -s -B %s\" failed: %s.\n",
						progname, progname, path,
						strerror(errno));
			}
			_exit(1);

		default:
			/* Parent */
			dist_node[i].io = io[1];
			dist_node[i].err = err[1];
			ret = close(io[0]);
			assert(0 <= ret);
			ret = close(err[0]);
			assert(0 <= ret);

			sprintf(buf, "%u", i);
			ret = write(dist_node[i].io, buf, sizeof(buf));
			if (ret != sizeof(buf)) {
				assert(0); /* FIXME */
			}

			ret = read(dist_node[i].io, buf, sizeof(buf));
			if (ret != sizeof(buf)) {
				assert(0); /* FIXME */
			}
			dist_node[i].port = atoi(buf);
			assert(1024 <= dist_node[i].port
			    && dist_node[i].port < 0x10000);

			/* proc_register(...); FIXME */
			io_register(err[1], (void *) 1, dist_err);
			break;
		}
	}

	dist_node[dist_node_count].name = strdup(name);
	assert(dist_node[dist_node_count].name);
	dist_node[dist_node_count].user = strdup(user);
	assert(dist_node[dist_node_count].user);
	dist_node[dist_node_count].host = strdup(host);
	assert(dist_node[dist_node_count].host);
	dist_node[dist_node_count].path = strdup(path);
	assert(dist_node[dist_node_count].path);
	return dist_node_count++;
}

int
dist_node_destroy(unsigned int id)
{
	/* FIXME */
	return 0;
}

void
dist_init(void)
{
	int ret;

	if (! slave) {
		/*
		 * Master
		 */
		struct passwd *pw;

		dist_node_count = 0;

		/*
		 * Get Default User
		 */
		pw = getpwuid(getuid());
		assert(pw);

		strcpy(def_user, pw->pw_name);

		/*
		 * Get Default Host
		 */
		ret = gethostname(def_host, sizeof(def_host));
		assert(0 <= ret);

		/*
		 * Get Default Path
		 */
		strcpy(def_path, "node.def");

	} else {
		/*
		 * Slave
		 */
		char buf[32];
		struct sockaddr_in addr;

		/* Get node number. */
		ret = read(0, buf, sizeof(buf));
		if (ret != sizeof(buf)) {
			assert(0); /* FIXME */
		}
		myself = atoi(buf);
		assert(1 <= myself
		    && myself < sizeof(dist_node) / sizeof(dist_node[0]));

		/* Connection to node 0 is established. */
		dist_node[0].io = 0;

		/* Create socket. */
		dist_node[myself].fd = socket(PF_INET, SOCK_STREAM, 0);
		assert(0 < dist_node[myself].fd);

		/* Set TCP_NODELAY option (see "man 7 tcp"). */
		/* FIXME */

		for (dist_node[myself].port = 1024;
		    ;
		    dist_node[myself].port++) {
			memset(&addr, 0, sizeof(addr));
			addr.sin_family = AF_INET;
			addr.sin_port = htons(dist_node[myself].port);
			addr.sin_addr.s_addr = INADDR_ANY;

			ret = bind(dist_node[myself].fd,
				(const struct sockaddr *) &addr, sizeof(addr));
			if (0 <= ret) {
				break;
			}
		}

		ret = listen(dist_node[myself].fd, 64);
		assert(0 <= ret);

		/* Send port number. */
		sprintf(buf, "%u", dist_node[myself].port);
		ret = write(1, buf, sizeof(buf));
		if (ret != sizeof(buf)) {
			assert(0); /* FIXME */
		}

		fprintf(stderr, "Running as node %u port %u.\n",
				myself, dist_node[myself].port);
	}
}

void
dist_exit(void)
{
	int ret;

	if (! slave) {
		/*
		 * Master
		 */
		/* FIXME */

	} else {
		/*
		 * Slave
		 */
		ret = close(dist_node[myself].fd);
		assert(0 <= ret);
	}
}
