/*
 * $Id: sig_usb_bus.c,v 1.26 2012-07-05 21:09:31 potyra Exp $
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#include "sig_usb_bus.h"

int
sig_usb_bus_reset_set(
	struct sig_usb_bus_main *sig,
	void *s,
	int val
)
{
	unsigned int nr;

	for (nr = 0; nr < sig->member_count; nr++) {
		if (! sig->member[nr].f
		 || ! sig->member[nr].f->reset_set
		 || sig->member[nr].s == s) {
			continue;
		}
		sig->member[nr].f->reset_set(sig->member[nr].s, val);
	}
	return 0;
}

int
sig_usb_bus_speed_set(
	struct sig_usb_bus_main *sig,
	void *s,
	int val
)
{
	unsigned int nr;

	for (nr = 0; nr < sig->member_count; nr++) {
		if (! sig->member[nr].f
		 || ! sig->member[nr].f->speed_set
		 || sig->member[nr].s == s) {
			continue;
		}
		sig->member[nr].f->speed_set(sig->member[nr].s, val);
	}
	return 0;
}

int
sig_usb_bus_send_token(
	struct sig_usb_bus_main *sig,
	void *s,
	int pid,
	int addr,
	int endp
)
{
	unsigned int nr;

	for (nr = 0; nr < sig->member_count; nr++) {
		if (! sig->member[nr].f
		 || ! sig->member[nr].f->recv_token
		 || sig->member[nr].s == s) {
			continue;
		}
		sig->member[nr].f->recv_token(sig->member[nr].s,
				pid, addr, endp);
	}
	return 0;
}

int
sig_usb_bus_send_sof(
	struct sig_usb_bus_main *sig,
	void *s,
	int frame_num
)
{
	unsigned int nr;

	for (nr = 0; nr < sig->member_count; nr++) {
		if (! sig->member[nr].f
		 || ! sig->member[nr].f->recv_sof
		 || sig->member[nr].s == s) {
			continue;
		}
		sig->member[nr].f->recv_sof(sig->member[nr].s, frame_num);
	}
	return 0;
}

int
sig_usb_bus_send_data(
	struct sig_usb_bus_main *sig,
	void *s,
	int pid,
	unsigned int length,
	uint8_t *data,
	uint16_t crc16
)
{
	unsigned int nr;

	for (nr = 0; nr < sig->member_count; nr++) {
		if (! sig->member[nr].f
		 || ! sig->member[nr].f->recv_data
		 || sig->member[nr].s == s) {
			continue;
		}
		sig->member[nr].f->recv_data(sig->member[nr].s,
				pid, length, data, crc16);
	}
	return 0;
}

int
sig_usb_bus_send_handshake(struct sig_usb_bus_main *sig, void *s, int pid)
{
	unsigned int nr;

	for (nr = 0; nr < sig->member_count; nr++) {
		if (! sig->member[nr].f
		 || ! sig->member[nr].f->recv_handshake
		 || sig->member[nr].s == s) {
			continue;
		}
		sig->member[nr].f->recv_handshake(sig->member[nr].s, pid);
	}
	return 0;
}

void
sig_usb_bus_main_connect(
	struct sig_usb_bus_main *sig,
	void *s,
	const struct sig_usb_bus_main_funcs *f
)
{
	assert(sig);
	assert(sig->type == SIG_GEN_USB_BUS_MAIN);
	assert(sig->member_count
			< sizeof(sig->member) / sizeof(sig->member[0]));

	sig->member[sig->member_count].s = s;
	sig->member[sig->member_count].f = f;
	sig->member_count++;
}

static void
sig_usb_bus_main_s0_reset_set(void *_f, int val)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_reset_set(f->s1, f, val);
}

static void
sig_usb_bus_main_s0_speed_set(void *_f, int val)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_speed_set(f->s1, f, val);
}

static void
sig_usb_bus_main_s0_recv_token(void *_f, int pid, int addr, int endp)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_send_token(f->s1, f, pid, addr, endp);
}

static void
sig_usb_bus_main_s0_recv_sof(void *_f, int frame_num)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_send_sof(f->s1, f, frame_num);
}

static void
sig_usb_bus_main_s0_recv_data(
	void *_f,
	int pid,
	unsigned int length,
	uint8_t *data,
	uint16_t crc16
)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_send_data(f->s1, f, pid, length, data, crc16);
}

static void
sig_usb_bus_main_s0_recv_handshake(void *_f, int pid)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_send_handshake(f->s1, f, pid);
}

static void
sig_usb_bus_main_s1_reset_set(void *_f, int val)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_reset_set(f->s0, f, val);
}

static void
sig_usb_bus_main_s1_speed_set(void *_f, int val)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_speed_set(f->s0, f, val);
}

static void
sig_usb_bus_main_s1_recv_token(void *_f, int pid, int addr, int endp)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_send_token(f->s0, f, pid, addr, endp);
}

static void
sig_usb_bus_main_s1_recv_sof(void *_f, int frame_num)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_send_sof(f->s0, f, frame_num);
}

static void
sig_usb_bus_main_s1_recv_data(
	void *_f,
	int pid,
	unsigned int length,
	uint8_t *data,
	uint16_t crc16
)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_send_data(f->s0, f, pid, length, data, crc16);
}

static void
sig_usb_bus_main_s1_recv_handshake(void *_f, int pid)
{
	struct sig_usb_bus_main_merge *f
			= (struct sig_usb_bus_main_merge *) _f;

	sig_usb_bus_send_handshake(f->s0, f, pid);
}

struct sig_usb_bus_main_merge *
sig_usb_bus_main_merge(
	struct sig_usb_bus_main *s0,
	struct sig_usb_bus_main *s1
)
{
	static const struct sig_usb_bus_main_funcs s0_funcs = {
		.reset_set = sig_usb_bus_main_s0_reset_set,
		.speed_set = sig_usb_bus_main_s0_speed_set,
		.recv_token = sig_usb_bus_main_s0_recv_token,
		.recv_sof = sig_usb_bus_main_s0_recv_sof,
		.recv_data = sig_usb_bus_main_s0_recv_data,
		.recv_handshake = sig_usb_bus_main_s0_recv_handshake,
	};
	static const struct sig_usb_bus_main_funcs s1_funcs = {
		.reset_set = sig_usb_bus_main_s1_reset_set,
		.speed_set = sig_usb_bus_main_s1_speed_set,
		.recv_token = sig_usb_bus_main_s1_recv_token,
		.recv_sof = sig_usb_bus_main_s1_recv_sof,
		.recv_data = sig_usb_bus_main_s1_recv_data,
		.recv_handshake = sig_usb_bus_main_s1_recv_handshake,
	};
	struct sig_usb_bus_main_merge *m;

	m = shm_alloc(sizeof(*m));
	assert(m);

	m->s0 = s0;
	sig_usb_bus_main_connect(s0, m, &s0_funcs);
	m->s1 = s1;
	sig_usb_bus_main_connect(s1, m, &s1_funcs);

	return m;
}

void
sig_usb_bus_main_split(struct sig_usb_bus_main_merge *m)
{
	fixme();
}

struct sig_usb_bus_main *
sig_usb_bus_main_create(const char *name)
{
	struct sig_usb_bus_main *sig;

	sig = shm_alloc(sizeof(*sig));
	assert(sig);

	sig->type = SIG_GEN_USB_BUS_MAIN;
	sig->member_count = 0;

	return sig;
}

void
sig_usb_bus_main_destroy(struct sig_usb_bus_main *sig)
{
	assert(sig);
	assert(sig->type == SIG_GEN_USB_BUS_MAIN);

	shm_free(sig);
}

void
sig_usb_bus_main_suspend(struct sig_usb_bus_main *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_suspend(b, size, fSig);
}

void
sig_usb_bus_main_resume(struct sig_usb_bus_main *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_resume(b, size, fSig);
}

struct sig_usb_bus *
sig_usb_bus_create(const char *name)
{
	struct sig_usb_bus *c;
	char n[1000];

	c = shm_alloc(sizeof(*c));
	assert(c);

	c->type = SIG_GEN_USB_BUS;

	sprintf(n, "%s-power", name);
	c->power = sig_boolean_create(n);
	sprintf(n, "%s-bus", name);
	c->bus = sig_usb_bus_main_create(n);

	return c;
}

void
sig_usb_bus_destroy(struct sig_usb_bus *sig)
{
	assert(sig);
	assert(sig->type == SIG_GEN_USB_BUS);

	sig_boolean_destroy(sig->power);
	sig_usb_bus_main_destroy(sig->bus);

	shm_free(sig);
}

void
sig_usb_bus_suspend(struct sig_usb_bus *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_suspend(b, size, fSig);
	
	sig_boolean_suspend(b->power, fSig);
	sig_usb_bus_main_suspend(b->bus, fSig);
}

void
sig_usb_bus_resume(struct sig_usb_bus *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_resume(b, size, fSig);
	
	sig_boolean_resume(b->power, fSig);
	sig_usb_bus_main_resume(b->bus, fSig);
}
