/***************************************************************************
                               charmm_long.cpp
                             -------------------
                            W. Michael Brown (ORNL)

  Class for acceleration of the charmm/coul/long pair style.

 __________________________________________________________________________
    This file is part of the LAMMPS Accelerator Library (LAMMPS_AL)
 __________________________________________________________________________

    begin                :
    email                : brownw@ornl.gov
 ***************************************************************************/

#if defined(USE_OPENCL)
#include "charmm_long_cl.h"
#elif defined(USE_CUDART)
const char *charmm_long=0;
#else
#include "charmm_long_cubin.h"
#endif

#include "lal_charmm_long.h"
#include <cassert>
namespace LAMMPS_AL {
#define CHARMMLongT CHARMMLong<numtyp, acctyp>

extern Device<PRECISION,ACC_PRECISION> device;

template <class numtyp, class acctyp>
CHARMMLongT::CHARMMLong() : BaseCharge<numtyp,acctyp>(),
                                    _allocated(false) {
}

template <class numtyp, class acctyp>
CHARMMLongT::~CHARMMLong() {
  clear();
}

template <class numtyp, class acctyp>
int CHARMMLongT::bytes_per_atom(const int max_nbors) const {
  return this->bytes_per_atom_atomic(max_nbors);
}

template <class numtyp, class acctyp>
int CHARMMLongT::init(const int ntypes, double host_cut_bothsq, double **host_lj1,
                      double **host_lj2, double **host_lj3, double **host_lj4,
                      double ** /*host_offset*/, double *host_special_lj, const int nlocal,
                      const int nall, const int max_nbors, const int maxspecial,
                      const double cell_size, const double gpu_split, FILE *_screen,
                      double host_cut_ljsq, const double host_cut_coulsq,
                      double *host_special_coul, const double qqrd2e, const double g_ewald,
                      const double cut_lj_innersq, const double denom_lj, double **epsilon,
                      double **sigma, const bool mix_arithmetic) {
  int success;
  success=this->init_atomic(nlocal,nall,max_nbors,maxspecial,cell_size,gpu_split,
                            _screen,charmm_long,"k_charmm_long");
  if (success!=0)
    return success;

  // If atom type constants fit in shared memory use fast kernel
  int lj_types=ntypes;
  shared_types=false;
  int max_bio_shared_types=this->device->max_bio_shared_types();
  if (this->_block_bio_size>=64 && mix_arithmetic &&
      lj_types<=max_bio_shared_types)
    shared_types=true;
  _lj_types=lj_types;

  // Allocate a host write buffer for data initialization
  int h_size=lj_types*lj_types;
  if (h_size<max_bio_shared_types)
    h_size=max_bio_shared_types;
  UCL_H_Vec<numtyp> host_write(h_size*32,*(this->ucl_device),
                               UCL_WRITE_ONLY);
  for (int i=0; i<h_size*32; i++)
    host_write[i]=0.0;

  lj1.alloc(lj_types*lj_types,*(this->ucl_device),UCL_READ_ONLY);
  this->atom->type_pack4(ntypes,lj_types,lj1,host_write,host_lj1,host_lj2,
                         host_lj3,host_lj4);

  if (shared_types) {
    ljd.alloc(max_bio_shared_types,*(this->ucl_device),UCL_READ_ONLY);
    this->atom->self_pack2(ntypes,ljd,host_write,epsilon,sigma);
  }

  sp_lj.alloc(8,*(this->ucl_device),UCL_READ_ONLY);
  for (int i=0; i<4; i++) {
    host_write[i]=host_special_lj[i];
    host_write[i+4]=host_special_coul[i];
  }
  ucl_copy(sp_lj,host_write,8,false);

  _cut_bothsq = host_cut_bothsq;
  _cut_coulsq = host_cut_coulsq;
  _cut_ljsq = host_cut_ljsq;
  _cut_lj_innersq = cut_lj_innersq;
  _qqrd2e=qqrd2e;
  _g_ewald=g_ewald;
  _denom_lj=denom_lj;

  _allocated=true;
  this->_max_bytes=lj1.row_bytes()+ljd.row_bytes()+sp_lj.row_bytes();
  return 0;
}

template <class numtyp, class acctyp>
void CHARMMLongT::clear() {
  if (!_allocated)
    return;
  _allocated=false;

  lj1.clear();
  ljd.clear();
  sp_lj.clear();
  this->clear_atomic();
}

template <class numtyp, class acctyp>
double CHARMMLongT::host_memory_usage() const {
  return this->host_memory_usage_atomic()+sizeof(CHARMMLong<numtyp,acctyp>);
}

// ---------------------------------------------------------------------------
// Calculate energies, forces, and torques
// ---------------------------------------------------------------------------
template <class numtyp, class acctyp>
int CHARMMLongT::loop(const int eflag, const int vflag) {
  // Compute the block size and grid size to keep all cores busy
  const int BX=this->_block_bio_size;
  int GX=static_cast<int>(ceil(static_cast<double>(this->ans->inum())/
                               (BX/this->_threads_per_atom)));

  int ainum=this->ans->inum();
  int nbor_pitch=this->nbor->nbor_pitch();
  this->time_pair.start();
  if (shared_types) {
    this->k_pair_sel->set_size(GX,BX);
    this->k_pair_sel->run(&this->atom->x, &ljd, &sp_lj,
                          &this->nbor->dev_nbor, &this->_nbor_data->begin(),
                          &this->ans->force, &this->ans->engv, &eflag,
                          &vflag, &ainum, &nbor_pitch, &this->atom->q,
                          &_cut_coulsq, &_qqrd2e, &_g_ewald, &_denom_lj,
                          &_cut_bothsq, &_cut_ljsq, &_cut_lj_innersq,
                          &this->_threads_per_atom);
  } else {
    this->k_pair.set_size(GX,BX);
    this->k_pair.run(&this->atom->x, &lj1, &_lj_types, &sp_lj,
                     &this->nbor->dev_nbor, &this->_nbor_data->begin(),
                     &this->ans->force, &this->ans->engv, &eflag, &vflag,
                     &ainum, &nbor_pitch, &this->atom->q,
                     &_cut_coulsq, &_qqrd2e, &_g_ewald, &_denom_lj,
                     &_cut_bothsq, &_cut_ljsq, &_cut_lj_innersq,
                     &this->_threads_per_atom);
  }
  this->time_pair.stop();
  return GX;
}

template class CHARMMLong<PRECISION,ACC_PRECISION>;
}
