/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.lucene.search;

import org.apache.lucene.index.Term;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.automaton.Automata;
import org.apache.lucene.util.automaton.Automaton;

/**
 * A Query that matches documents within an range of terms.
 *
 * <p>This query matches the documents looking for terms that fall into the supplied range according
 * to {@link BytesRef#compareTo(BytesRef)}.
 *
 * <p><b>NOTE</b>: {@link TermRangeQuery} performs significantly slower than {@link PointRangeQuery
 * point-based ranges} as it needs to visit all terms that match the range and merges their matches.
 *
 * <p>This query uses the {@link MultiTermQuery#CONSTANT_SCORE_BLENDED_REWRITE} rewrite method.
 *
 * @since 2.9
 */
public class TermRangeQuery extends AutomatonQuery {
  private final BytesRef lowerTerm;
  private final BytesRef upperTerm;
  private final boolean includeLower;
  private final boolean includeUpper;

  /**
   * Constructs a query selecting all terms greater/equal than <code>lowerTerm</code> but less/equal
   * than <code>upperTerm</code>.
   *
   * <p>If an endpoint is null, it is said to be "open". Either or both endpoints may be open. Open
   * endpoints may not be exclusive (you can't select all but the first or last term without
   * explicitly specifying the term to exclude.)
   *
   * @param field The field that holds both lower and upper terms.
   * @param lowerTerm The term text at the lower end of the range
   * @param upperTerm The term text at the upper end of the range
   * @param includeLower If true, the <code>lowerTerm</code> is included in the range.
   * @param includeUpper If true, the <code>upperTerm</code> is included in the range.
   */
  public TermRangeQuery(
      String field,
      BytesRef lowerTerm,
      BytesRef upperTerm,
      boolean includeLower,
      boolean includeUpper) {
    this(field, lowerTerm, upperTerm, includeLower, includeUpper, CONSTANT_SCORE_BLENDED_REWRITE);
  }

  /**
   * Constructs a query selecting all terms greater/equal than <code>lowerTerm</code> but less/equal
   * than <code>upperTerm</code>.
   *
   * <p>If an endpoint is null, it is said to be "open". Either or both endpoints may be open. Open
   * endpoints may not be exclusive (you can't select all but the first or last term without
   * explicitly specifying the term to exclude.)
   *
   * @param field The field that holds both lower and upper terms.
   * @param lowerTerm The term text at the lower end of the range
   * @param upperTerm The term text at the upper end of the range
   * @param includeLower If true, the <code>lowerTerm</code> is included in the range.
   * @param includeUpper If true, the <code>upperTerm</code> is included in the range.
   * @param rewriteMethod the rewrite method to use when building the final query
   */
  public TermRangeQuery(
      String field,
      BytesRef lowerTerm,
      BytesRef upperTerm,
      boolean includeLower,
      boolean includeUpper,
      RewriteMethod rewriteMethod) {
    super(
        new Term(field, lowerTerm),
        toAutomaton(lowerTerm, upperTerm, includeLower, includeUpper),
        Integer.MAX_VALUE,
        true,
        rewriteMethod);
    this.lowerTerm = lowerTerm;
    this.upperTerm = upperTerm;
    this.includeLower = includeLower;
    this.includeUpper = includeUpper;
  }

  public static Automaton toAutomaton(
      BytesRef lowerTerm, BytesRef upperTerm, boolean includeLower, boolean includeUpper) {

    if (lowerTerm == null) {
      // makeBinaryInterval is more picky than we are:
      includeLower = true;
    }

    if (upperTerm == null) {
      // makeBinaryInterval is more picky than we are:
      includeUpper = true;
    }

    return Automata.makeBinaryInterval(lowerTerm, includeLower, upperTerm, includeUpper);
  }

  /** Factory that creates a new TermRangeQuery using Strings for term text. */
  public static TermRangeQuery newStringRange(
      String field,
      String lowerTerm,
      String upperTerm,
      boolean includeLower,
      boolean includeUpper) {
    return newStringRange(
        field, lowerTerm, upperTerm, includeLower, includeUpper, CONSTANT_SCORE_BLENDED_REWRITE);
  }

  /** Factory that creates a new TermRangeQuery using Strings for term text. */
  public static TermRangeQuery newStringRange(
      String field,
      String lowerTerm,
      String upperTerm,
      boolean includeLower,
      boolean includeUpper,
      RewriteMethod rewriteMethod) {
    BytesRef lower = lowerTerm == null ? null : new BytesRef(lowerTerm);
    BytesRef upper = upperTerm == null ? null : new BytesRef(upperTerm);
    return new TermRangeQuery(field, lower, upper, includeLower, includeUpper, rewriteMethod);
  }

  /** Returns the lower value of this range query */
  public BytesRef getLowerTerm() {
    return lowerTerm;
  }

  /** Returns the upper value of this range query */
  public BytesRef getUpperTerm() {
    return upperTerm;
  }

  /** Returns <code>true</code> if the lower endpoint is inclusive */
  public boolean includesLower() {
    return includeLower;
  }

  /** Returns <code>true</code> if the upper endpoint is inclusive */
  public boolean includesUpper() {
    return includeUpper;
  }

  /** Prints a user-readable version of this query. */
  @Override
  public String toString(String field) {
    StringBuilder buffer = new StringBuilder();
    if (!getField().equals(field)) {
      buffer.append(getField());
      buffer.append(":");
    }
    buffer.append(includeLower ? '[' : '{');
    // TODO: all these toStrings for queries should just output the bytes, it might not be UTF-8!
    buffer.append(
        lowerTerm != null
            ? ("*".equals(Term.toString(lowerTerm)) ? "\\*" : Term.toString(lowerTerm))
            : "*");
    buffer.append(" TO ");
    buffer.append(
        upperTerm != null
            ? ("*".equals(Term.toString(upperTerm)) ? "\\*" : Term.toString(upperTerm))
            : "*");
    buffer.append(includeUpper ? ']' : '}');
    return buffer.toString();
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = super.hashCode();
    result = prime * result + (includeLower ? 1231 : 1237);
    result = prime * result + (includeUpper ? 1231 : 1237);
    result = prime * result + ((lowerTerm == null) ? 0 : lowerTerm.hashCode());
    result = prime * result + ((upperTerm == null) ? 0 : upperTerm.hashCode());
    return result;
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) return true;
    if (!super.equals(obj)) return false;
    if (getClass() != obj.getClass()) return false;
    TermRangeQuery other = (TermRangeQuery) obj;
    if (includeLower != other.includeLower) return false;
    if (includeUpper != other.includeUpper) return false;
    if (lowerTerm == null) {
      if (other.lowerTerm != null) return false;
    } else if (!lowerTerm.equals(other.lowerTerm)) return false;
    if (upperTerm == null) {
      if (other.upperTerm != null) return false;
    } else if (!upperTerm.equals(other.upperTerm)) return false;
    return true;
  }
}
