#include <iostream>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <thread>
#include <functional>
#include <atomic>
#include <condition_variable>

#include "../gig.h"
#include "../helper.h"

#if !NO_MAIN || !defined(TEST_ASSERT)
static int iErrors = 0;
static int iChecks = 0;
#endif

#ifndef TEST_ASSERT
# define TEST_ASSERT(...) iChecks++; if (!(__VA_ARGS__)) ++iErrors
#endif

#ifndef TEST_VERIFY
# define TEST_VERIFY(...) do {                                               \
    if (!(__VA_ARGS__)) {                                                    \
        fprintf(stderr, "\n[ERROR] %s() failed:\n\n", __FUNCTION__);         \
        fprintf(stderr, "Violated expectation for this failure was:\n\n  "   \
                #__VA_ARGS__ "\n\n  [%s at line %d]\n\n",                    \
                lastPathComponent(__FILE__).c_str(), __LINE__);              \
    }                                                                        \
    TEST_ASSERT(__VA_ARGS__);                                                \
} while (false);
#endif

using namespace std;

// file names of the Gigasampler files we are going to create for these tests
#define TEST_GIG_FILE_NAME "foo.gig"
#define TEST_GIG_FILE_NAME2 "foo2.gig"

// four stupid little sample "waves"
// (each having three sample points length, 16 bit depth, mono)
int16_t sampleData1[] = { 1, 2, 3 };
int16_t sampleData2[] = { 4, 5, 6 };
int16_t sampleData3[] = { 7, 8, 9 };
int16_t sampleData4[] = { 10,11,12 };

class PubFile : public gig::File {
public:
    using gig::File::VerifySampleChecksumTable;
};


/////////////////////////////////////////////////////////////////////////////
// The actual test cases (in order) ...

// 1. Run) create a new Gigasampler file from scratch
static void test_createNewGigFile() {
    try {
        // create an empty Gigasampler file
        gig::File file;
        // we give it an internal name, not mandatory though
        file.pInfo->Name = "Foo Gigasampler File";

        // create four samples
        gig::Sample* pSample1 = file.AddSample();
        gig::Sample* pSample2 = file.AddSample();
        gig::Sample* pSample3 = file.AddSample();
        gig::Sample* pSample4 = file.AddSample();
        // give those samples a name (not mandatory)
        pSample1->pInfo->Name = "Foo Sample 1";
        pSample2->pInfo->Name = "Foo Sample 2";
        pSample3->pInfo->Name = "Foo Sample 3";
        pSample4->pInfo->Name = "Foo Sample 4";
        // set meta information for those samples
        pSample1->Channels = 1; // mono
        pSample1->BitDepth = 16; // 16 bits
        pSample1->FrameSize = 16/*bitdepth*/ / 8/*1 byte are 8 bits*/ * 1/*mono*/;
        pSample1->SamplesPerSecond = 44100;
        pSample2->Channels = 1; // mono
        pSample2->BitDepth = 16; // 16 bits
        pSample2->FrameSize = 16 / 8 * 1;
        pSample2->SamplesPerSecond = 44100;
        pSample3->Channels = 1; // mono
        pSample3->BitDepth = 16; // 16 bits
        pSample3->FrameSize = 16 / 8 * 1;
        pSample3->SamplesPerSecond = 44100;
        pSample4->Channels = 1; // mono
        pSample4->BitDepth = 16; // 16 bits
        pSample4->FrameSize = 16 / 8 * 1;
        pSample4->SamplesPerSecond = 44100;
        // resize those samples to a length of three sample points
        // (again: _sample_points_ NOT bytes!) which is the length of our
        // ficticious samples from above. after the Save() call below we can
        // then directly write our sample data to disk by using the Write()
        // method, that is without having to load all the sample data into
        // RAM. for large instruments / .gig files this is definitely the way
        // to go
        pSample1->Resize(3);
        pSample2->Resize(3);
        pSample3->Resize(3);
        pSample4->Resize(3);

        // create four instruments
        gig::Instrument* pInstrument1 = file.AddInstrument();
        gig::Instrument* pInstrument2 = file.AddInstrument();
        gig::Instrument* pInstrument3 = file.AddInstrument();
        gig::Instrument* pInstrument4 = file.AddInstrument();
        // give them a name (not mandatory)
        pInstrument1->pInfo->Name = "Foo Instrument 1";
        pInstrument2->pInfo->Name = "Foo Instrument 2";
        pInstrument3->pInfo->Name = "Foo Instrument 3";
        pInstrument4->pInfo->Name = "Foo Instrument 4";

        // create one region for each instrument
        // in this example we do not add a dimension, so
        // every region will have exactly one DimensionRegion
        // also we assign a sample to each dimension region
        gig::Region* pRegion = pInstrument1->AddRegion();
        pRegion->SetSample(pSample1);
        pRegion->KeyRange.low  = 0;
        pRegion->KeyRange.high = 1;
        pRegion->VelocityRange.low  = 0;
        pRegion->VelocityRange.high = 1;
        pRegion->KeyGroup = 0;
        pRegion->pDimensionRegions[0]->pSample = pSample1;

        pRegion = pInstrument2->AddRegion();
        pRegion->SetSample(pSample2);
        pRegion->KeyRange.low  = 1;
        pRegion->KeyRange.high = 2;
        pRegion->VelocityRange.low  = 1;
        pRegion->VelocityRange.high = 2;
        pRegion->KeyGroup = 1;
        pRegion->pDimensionRegions[0]->pSample = pSample2;

        pRegion = pInstrument3->AddRegion();
        pRegion->SetSample(pSample3);
        pRegion->KeyRange.low  = 2;
        pRegion->KeyRange.high = 3;
        pRegion->VelocityRange.low  = 2;
        pRegion->VelocityRange.high = 3;
        pRegion->KeyGroup = 2;
        pRegion->pDimensionRegions[0]->pSample = pSample3;

        pRegion = pInstrument4->AddRegion();
        pRegion->SetSample(pSample4);
        pRegion->KeyRange.low  = 3;
        pRegion->KeyRange.high = 4;
        pRegion->VelocityRange.low  = 3;
        pRegion->VelocityRange.high = 4;
        pRegion->KeyGroup = 3;
        pRegion->pDimensionRegions[0]->pSample = pSample4;

        // save file ("physically") as of now
        file.Save(TEST_GIG_FILE_NAME);
    } catch (RIFF::Exception e) {
        std::cerr << "\nCould not create a new Gigasampler file from scratch:\n" << std::flush;
        e.PrintMessage();
        throw e; // stop further tests
    }
}

// 2. Run) test if the newly created Gigasampler file exists & can be opened
static void test_openCreatedGigFile() {
    // try to open previously created Gigasampler file
    try {
        RIFF::File riff(TEST_GIG_FILE_NAME);
        gig::File file(&riff);
    } catch (RIFF::Exception e) {
        std::cerr << "\nCould not open newly created Gigasampler file:\n" << std::flush;
        e.PrintMessage();
        throw e; // stop further tests
    }
}

// 3. Run) test if the articulation information of the newly created Gigasampler file were correctly written
static void test_articulationsOfCreatedGigFile() {
    try {
        // open previously created Gigasampler file
        RIFF::File riff(TEST_GIG_FILE_NAME);
        gig::File file(&riff);
        // check global file information
        TEST_VERIFY(file.pInfo);
        TEST_VERIFY(file.pInfo->Name == "Foo Gigasampler File");
        // check amount of instruments and samples
        TEST_VERIFY(file.Instruments == 4);
        TEST_VERIFY(file.CountInstruments() == 4);
        TEST_VERIFY(file.CountSamples() == 4);
        // check samples' meta information
        for (size_t s = 0; gig::Sample* pSample = file.GetSample(s); ++s) {
            TEST_VERIFY(pSample->pInfo);
            std::string sOughtToBe = "Foo Sample " + ToString(s+1);
            TEST_VERIFY(pSample->pInfo->Name == sOughtToBe);
            TEST_VERIFY(pSample->GetSize() == 3); // three sample points
            TEST_VERIFY(pSample->SamplesTotal == 3); // three sample points
            TEST_VERIFY(pSample->Channels == 1); // mono
            TEST_VERIFY(pSample->BitDepth == 16); // bit depth 16 bits
            TEST_VERIFY(pSample->FrameSize == 16 / 8 * 1);
            TEST_VERIFY(pSample->SamplesPerSecond == 44100);
        }
        // check instruments' meta information
        for (size_t i = 0; gig::Instrument* pInstrument = file.GetInstrument(i); ++i) {
            TEST_VERIFY(pInstrument->pInfo);
            std::string sOughtToBe = "Foo Instrument " + ToString(i+1);
            TEST_VERIFY(pInstrument->pInfo->Name == sOughtToBe);
            gig::Region* pRegion = pInstrument->GetRegionAt(0);
            TEST_VERIFY(pRegion);
            TEST_VERIFY(pRegion->Dimensions == 0);
            TEST_VERIFY(pRegion->DimensionRegions == 1);
            sOughtToBe = "Foo Sample " + ToString(i+1);
            TEST_VERIFY(pRegion->GetSample()->pInfo->Name == sOughtToBe);
            TEST_VERIFY(pRegion->KeyRange.low  == i);
            TEST_VERIFY(pRegion->KeyRange.high == i+1);
            TEST_VERIFY(pRegion->VelocityRange.low  == i);
            TEST_VERIFY(pRegion->VelocityRange.high == i+1);
            TEST_VERIFY(pRegion->KeyGroup  == i);
            const uint dimValues[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };
            gig::DimensionRegion* pDimensionRegion = pRegion->GetDimensionRegionByValue(dimValues);
            TEST_VERIFY(pDimensionRegion);
            TEST_VERIFY(pDimensionRegion->pSample->pInfo->Name == sOughtToBe);
        }
    } catch (RIFF::Exception e) {
        std::cerr << "\nThere was an exception while checking the articulation data of the newly created Gigasampler file:\n" << std::flush;
        e.PrintMessage();
        throw e; // stop further tests
    }
}

// 4. Run) try to write sample data to that newly created Gigasampler file
static void test_writeSamples() {
    try {
        // open previously created Gigasampler file (in read/write mode)
        RIFF::File riff(TEST_GIG_FILE_NAME);
        riff.SetMode(RIFF::stream_mode_read_write);
        gig::File file(&riff);
        // until this point we just wrote the articulation data to the .gig file
        // and prepared the .gig file for writing our 4 example sample data, so
        // now as the file exists physically and the 'samples' are already
        // of the correct size we can now write the actual samples' data by
        // directly writing them to disk
        gig::Sample* pSample1 = file.GetSample(0);
        gig::Sample* pSample2 = file.GetSample(1);
        gig::Sample* pSample3 = file.GetSample(2);
        gig::Sample* pSample4 = file.GetSample(3);
        TEST_VERIFY(pSample1);
        pSample1->Write(sampleData1, 3);
        TEST_VERIFY(pSample2);
        pSample2->Write(sampleData2, 3);
        TEST_VERIFY(pSample3);
        pSample3->Write(sampleData3, 3);
        TEST_VERIFY(pSample4);
        pSample4->Write(sampleData4, 3);
    } catch (RIFF::Exception e) {
        std::cerr << "\nCould not directly write samples to newly created Gigasampler file:\n" << std::flush;
        e.PrintMessage();
        throw e; // stop further tests
    }
}

// 5. Run) check the previously written samples' data
static void test_samplesData() {
    try {
        // open previously created Gigasampler file
        RIFF::File riff(TEST_GIG_FILE_NAME);
        gig::File file(&riff);
        // check samples' meta information
        gig::Sample* pSample1 = file.GetSample(0);
        gig::Sample* pSample2 = file.GetSample(1);
        gig::Sample* pSample3 = file.GetSample(2);
        gig::Sample* pSample4 = file.GetSample(3);
        TEST_VERIFY(pSample1);
        TEST_VERIFY(pSample2);
        TEST_VERIFY(pSample3);
        TEST_VERIFY(pSample4);
        gig::buffer_t sampleBuffer1 = pSample1->LoadSampleData();
        gig::buffer_t sampleBuffer2 = pSample2->LoadSampleData();
        gig::buffer_t sampleBuffer3 = pSample3->LoadSampleData();
        gig::buffer_t sampleBuffer4 = pSample4->LoadSampleData();
        TEST_VERIFY(sampleBuffer1.pStart);
        TEST_VERIFY(sampleBuffer2.pStart);
        TEST_VERIFY(sampleBuffer3.pStart);
        TEST_VERIFY(sampleBuffer4.pStart);
        TEST_VERIFY(sampleBuffer1.Size == pSample1->FrameSize * 3); // three sample points length
        TEST_VERIFY(sampleBuffer2.Size == pSample2->FrameSize * 3); // three sample points length
        TEST_VERIFY(sampleBuffer3.Size == pSample3->FrameSize * 3); // three sample points length
        TEST_VERIFY(sampleBuffer4.Size == pSample4->FrameSize * 3); // three sample points length
        // check samples' PCM data
        TEST_VERIFY(memcmp(sampleBuffer1.pStart, sampleData1, 3) == 0);
        TEST_VERIFY(memcmp(sampleBuffer2.pStart, sampleData2, 3) == 0);
        TEST_VERIFY(memcmp(sampleBuffer3.pStart, sampleData3, 3) == 0);
        TEST_VERIFY(memcmp(sampleBuffer4.pStart, sampleData4, 3) == 0);
    } catch (RIFF::Exception e) {
        std::cerr << "\nThere was an exception while checking the written samples' data:\n" << std::flush;
        e.PrintMessage();
        throw e; // stop further tests
    }
}

// 6. Run) ask libgig to verify samples by their written checksums
static void test_verifySampleChecksums() {
    try {
        // open previously created Gigasampler file
        RIFF::File riff(TEST_GIG_FILE_NAME);
        gig::File file(&riff);
        PubFile& gig = (PubFile&) file;
        // verify checksum table itself
        TEST_VERIFY(gig.VerifySampleChecksumTable());
        // verify samples' checksums
        for (size_t s = 0; gig::Sample* sample = file.GetSample(s); ++s) {
            TEST_VERIFY(sample->VerifyWaveData());
        }
    } catch (RIFF::Exception e) {
        std::cerr << "\nThere was an exception while trying to verify samples' checksums:\n" << std::flush;
        e.PrintMessage();
        throw e; // stop further tests
    }
}

// 7. Run) open .gig file and save it as another .gig file
static void test_duplicateGigFile() {
    try {
        // open previously created gig file
        RIFF::File riff(TEST_GIG_FILE_NAME);
        gig::File file(&riff);
        // make sure file is fully loaded
        file.GetInstrument(0);
        // save as another gig file
        file.Save(TEST_GIG_FILE_NAME2);
    } catch (RIFF::Exception e) {
        std::cerr << "\nCould not duplicate Gigasampler file:\n" << std::flush;
        e.PrintMessage();
        throw e; // stop further tests
    }
}

// 8. Run) verify duplicated .gig file
static void test_verifyDuplicatedGigFile() {
    try {
        // open duplicated Gigasampler file
        RIFF::File riff(TEST_GIG_FILE_NAME2);
        gig::File file(&riff);
        PubFile& gig = (PubFile&) file;
        // verify checksum table itself
        TEST_VERIFY(gig.VerifySampleChecksumTable());
        // verify samples' checksums
        for (size_t s = 0; gig::Sample* sample = file.GetSample(s); ++s) {
            TEST_VERIFY(sample->VerifyWaveData());
        }
    } catch (RIFF::Exception e) {
        std::cerr << "\nThere was an exception while trying to verify duplicated gig file:\n" << std::flush;
        e.PrintMessage();
        throw e; // stop further tests
    }
}

// 9. Run) open .gig file on one thread, save on another thread
static void test_multithreadedLoadSave() {
    RIFF::File* sharedRiff = NULL;
    gig::File* sharedFile = NULL;
    std::mutex m;
    std::condition_variable cv;
    bool thread1Completed = false;
    std::atomic<bool> loaded(false);
    std::atomic<bool> saved(false);
    std::atomic<int> exceptions(0);
    std::thread::id threadIDs[2];

    std::thread thread1([&](){
        // so that we can verify that we really used 2 different threads
        threadIDs[0] = std::this_thread::get_id();
        try {
            // open previously created gig file
            sharedRiff = new RIFF::File(TEST_GIG_FILE_NAME);
            sharedFile = new gig::File(sharedRiff);
            // make sure file is fully loaded
            sharedFile->GetInstrument(0);
            // flag to 2nd thread that file is ready for usage
            loaded.store(true);
        } catch (RIFF::Exception e) {
            std::cerr << "\nCould not load gig file on thread 1:\n" << std::flush;
            e.PrintMessage();
            exceptions.fetch_add(1);
        }
        // notify 2nd thread that thread 1 completed
        {
            std::lock_guard<std::mutex> lock(m);
            thread1Completed = true;
        }
        cv.notify_one();
    });

    std::thread thread2([&](){
        // so that we can verify that we really used 2 different threads
        threadIDs[1] = std::this_thread::get_id();
        // wait for 1st thread to complete
        {
            std::unique_lock<std::mutex> lock(m);
            cv.wait(lock, [&] { return thread1Completed; });
        }
        // abort this 2nd thread if 1st thread failed loading the gig file
        if (!loaded.load()) {
            std::cerr << "\nThread 1 didn't load gig file, aborting thread 2.\n" << std::flush;
            return;
        }
        try {
            // save as another gig file
            sharedFile->Save(TEST_GIG_FILE_NAME2);
            // notify main thread that file was saved
            saved.store(true);
        } catch (RIFF::Exception e) {
            std::cerr << "\nCould not save gig file on thread 2:\n" << std::flush;
            e.PrintMessage();
            exceptions.fetch_add(1);
        }
    });

    // wait for both threads to complete
    thread1.join();
    thread2.join();

    TEST_VERIFY(!exceptions.load());
    TEST_VERIFY(loaded.load());
    TEST_VERIFY(saved.load());
    TEST_VERIFY(threadIDs[0] != threadIDs[1]);

    delete sharedFile;
    delete sharedRiff;
}

// 10. Run) verify (previously multi-threaded) duplicated .gig file
static void test_verifyMultithreadedDuplicatedGigFile() {
    try {
        // open duplicated Gigasampler file
        RIFF::File riff(TEST_GIG_FILE_NAME2);
        gig::File file(&riff);
        PubFile& gig = (PubFile&) file;
        // verify checksum table itself
        TEST_VERIFY(gig.VerifySampleChecksumTable());
        // verify samples' checksums
        for (size_t s = 0; gig::Sample* sample = file.GetSample(s); ++s) {
            TEST_VERIFY(sample->VerifyWaveData());
        }
    } catch (RIFF::Exception e) {
        std::cerr << "\nThere was an exception while trying to verify multithreaded duplicated gig file:\n" << std::flush;
        e.PrintMessage();
        throw e; // stop further tests
    }
}

#if !NO_MAIN

int main() {
    printf("\n");

    int iExceptions = 0;
    try {
        test_createNewGigFile();
        test_openCreatedGigFile();
        test_articulationsOfCreatedGigFile();
        test_writeSamples();
        test_samplesData();
        test_verifySampleChecksums();
        test_duplicateGigFile();
        test_verifyDuplicatedGigFile();
        test_multithreadedLoadSave();
        test_verifyMultithreadedDuplicatedGigFile();
    } catch (...) {
        iExceptions++;
    }

    if (iErrors || iExceptions)
        printf("\n!!! FAILED !!!  [ There were %d errors, %d exceptions ]\n\n",
               iErrors, iExceptions);
    else
        printf("\nOK (%d checks)\n\n", iChecks);

    return iErrors + iExceptions;
}

#endif // !NO_MAIN
