// Copyright (C) 2025 EDF
// All Rights Reserved
// This code is published under the GNU Lesser General Public License (GNU LGPL)
#ifndef MESH1DGENERS_H
#define MESH1DGENERS_H
#include "geners/GenericIO.hh"
#include "StOpt/core/grids/Mesh1D.h"

/** \file Mesh1DGeners.h
 *  \brief Define non intrusive serialization with random access
*  \author Pierre Gruet
 */
/// specialize the ClassIdSpecialization template
/// so that a ClassId object can be associated with the class we want to
/// serialize.  The second argument is the version number.
///@{
gs_specialize_class_id(StOpt::Mesh1D, 1)
/// an external class
gs_declare_type_external(StOpt::Mesh1D)
///@}
namespace gs
{
//
/// \brief  This is how the specialization of GenericWriter should look like
//
template <class Stream, class State >
struct GenericWriter < Stream, State, StOpt::Mesh1D,
           Int2Type<IOTraits<int>::ISEXTERNAL> >
{
    inline static bool process(const StOpt::Mesh1D  &p_mesh, Stream &p_os,
                               State *, const bool p_processClassId)
    {
        // If necessary, serialize the class id
        static const ClassId current(ClassId::makeId<StOpt::Mesh1D >());
        const bool status = p_processClassId ? ClassId::makeId<StOpt::Mesh1D >().write(p_os) : true;
        // Serialize object data if the class id was successfully
        // written out
        if (status)
        {
	  write_pod(p_os, p_mesh.getXL());
	  write_pod(p_os, p_mesh.getXR());
	  write_pod(p_os, p_mesh.getVerticeL());
	  write_pod(p_os, p_mesh.getVerticeR());
       }
        // Return "true" on success, "false" on failure
        return status && !p_os.fail();
    }
};

/// \brief  And this is the specialization of GenericReader
//
template <class Stream, class State  >
struct GenericReader < Stream, State, StOpt::Mesh1D, Int2Type<IOTraits<int>::ISEXTERNAL> >
{
    inline static bool readIntoPtr(StOpt::Mesh1D  *&ptr, Stream &p_is,
                                   State *p_st, const bool p_processClassId)
    {

      // Make sure that the serialized class id is consistent with
      // the current one
      static const ClassId current(ClassId::makeId<StOpt::Mesh1D>());
      const ClassId &stored = p_processClassId ? ClassId(p_is, 1) : p_st->back();
      current.ensureSameId(stored);
      
      double xL;
      read_pod(p_is,&xL);
      double xR;
      read_pod(p_is,&xR);
      int verticeL;
      read_pod(p_is, &verticeL);
      int verticeR;
      read_pod(p_is, &verticeR);
      if (p_is.fail())
	// Return "false" on failure
	return false;
      //Build the object from the stored data
      if (ptr)
        {
	  *ptr = StOpt::Mesh1D(xL,xR,verticeL,verticeR);
        }
      else
	{
	   ptr = new StOpt::Mesh1D(xL,xR,verticeL,verticeR);
	}
      return true;
    }

    inline static bool process(StOpt::Mesh1D &s, Stream &is,
                               State *st, const bool p_processClassId)
    {
      // Simply convert reading by reference into reading by pointer
      StOpt::Mesh1D *ps = &s;
      return readIntoPtr(ps, is, st, p_processClassId);
    }
};
}


#endif
